/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "gui_mainwindow.h"
#include "ui_mainwindow.h"
#include "miscellaneous.h"
#include "serial_v2.h"
#include <iostream>
#include <math.h>
#include <QCoreApplication>
#include <QCursor>
#include <QDebug>
#include <QDir>
#include <QMessageBox>
#include <QNetworkInterface>
#include <QtSerialPort/QSerialPort>
#include <QTimer>
#include <QThread>



QT_USE_NAMESPACE

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent), ui(new Ui::MainWindow)
{
	ui->setupUi(this);
	firmware_version_requirement[0] = 1;
	firmware_version_requirement[1] = 6;
	firmware_version_requirement[2] = 4;
	firmware_version_requirement[3] = 0;
	toolkit_GUI_version = "1.4.18";

	qDebug() << "Toolkit GUI " << toolkit_GUI_version;
	ui->GUI_version_label_2->setText(toolkit_GUI_version);

	/* get the names of the serialports, as well as all the other startup values from config.txt */
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

#ifdef DEBUG
	// Help Menu items
	//
	// TODO:
	// Exact items still to be figured out.
	//
	listWidgetItem_system.setText("System");
	listWidgetItem_about.setText("About");
	listWidgetItem_license.setText("License");
	listWidgetItem_copyright.setText("Copyright");

	ui->listWidget->addItem(&listWidgetItem_system);
	ui->listWidget->addItem(&listWidgetItem_license);
	ui->listWidget->addItem(&listWidgetItem_copyright);
	ui->listWidget->addItem(&listWidgetItem_about);
#endif //DEBUG

	/* set up button groups, make it so only one button can be checked at a time */
	menu_buttonGroup.addButton(ui->menu_home_Button);
	menu_buttonGroup.addButton(ui->menu_ALL_Button);
	menu_buttonGroup.addButton(ui->menu_DLL_Button);
	menu_buttonGroup.addButton(ui->menu_sweep_Button);
	menu_buttonGroup.addButton(ui->menu_settings_Button);
	menu_buttonGroup.addButton(ui->menu_help_Button);
	menu_buttonGroup.addButton(ui->menu_power_Button);
	//menu_buttonGroup.addButton(ui->menu_PID_Button);
	menu_buttonGroup.addButton(ui->menu_PSU_Button);
	menu_buttonGroup.setExclusive(true);

	chain_mode_buttonGroup.addButton(ui->pushButton_clock_source_standalone_1);
	chain_mode_buttonGroup.addButton(ui->pushButton_clock_source_master_1);
	chain_mode_buttonGroup.addButton(ui->pushButton_clock_source_slave_inline_1);
	chain_mode_buttonGroup.addButton(ui->pushButton_clock_source_slave_end_1);
	chain_mode_buttonGroup.setExclusive(true);

	RCM_buttonGroup.addButton(ui->pushButton_remote_command_OFF_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_USB_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_TCP_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_RS232_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_RS485_1);
	RCM_buttonGroup.setExclusive(true);

	PWM_triggering_buttonGroup.addButton(ui->pushButton_PWM_triggering_free_running_1);
	PWM_triggering_buttonGroup.addButton(ui->pushButton_PWM_triggering_master_1);
	PWM_triggering_buttonGroup.addButton(ui->pushButton_PWM_triggering_slave_1);
	PWM_triggering_buttonGroup.setExclusive(true);

	Power_control_mode_buttonGroup.addButton(ui->pushButton_power_control_normal_1);
	Power_control_mode_buttonGroup.addButton(ui->pushButton_power_control_analog_1);
	Power_control_mode_buttonGroup.addButton(ui->pushButton_power_control_feedforward_1);
	Power_control_mode_buttonGroup.addButton(ui->pushButton_power_control_amplifier_1);
	Power_control_mode_buttonGroup.setExclusive(true);

	External_triggering_buttonGroup.addButton(ui->pushButton_external_triggering_OFF_1);
	External_triggering_buttonGroup.addButton(ui->pushButton_external_triggering_ON_1);
	External_triggering_buttonGroup.setExclusive(true);

	/* set maximum rows on the RCM text edit. */
	ui->settings_plainTextEdit->setMaximumBlockCount(100);

	/* hide the DLL frequency indicator by default */
	ui->label_DLL_frequency_lock_1->setVisible(false);
	ui->label_DLL_frequency_lock_2->setVisible(false);
	ui->label_DLL_frequency_lock_3->setVisible(false);

	/* Hide the notifications by default */
	ui->label_notification->setVisible(false);
	ui->label_error_message->setVisible(false);

	#ifndef DEBUG
	/* temporary disables */
	ui->menu_help_Button->setVisible(false);	//Empty atm, but useful for debug buttons.
	ui->menu_help_Button->setEnabled(false);
	#endif

	/* Declare the timers. */
	notification_timer = new QTimer(this);
	readings_timer = new QTimer(this);			//Timer for power readings, Status polling.

	/* If Linux OS + RPi HAT (ideally Raspberry Pi) -> prep to use GPIO pins for RPi HAT */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external = new GPIOClass(QString::number(6));
		GPIO_control_SGx = new GPIOClass(QString::number(12));
		GPIO_HAT_serial_mode = new GPIOClass(QString::number(4));
		GPIO_RS485_RW_flipper = new GPIOClass(QString::number(18));
		GPIO_LED_1 = new GPIOClass(QString::number(13));
		GPIO_interlock_out = new GPIOClass(QString::number(20));		//Provides 3.3V that goes through Front Panel button + interlocks and is read back by GPIO 19.
		GPIO_PSU_button_enable = new GPIOClass(QString::number(19));

		GPIO_control_external->setDirection("out");
		GPIO_control_SGx->setDirection("out");
		GPIO_HAT_serial_mode->setDirection("out");
		GPIO_RS485_RW_flipper->setDirection("out");
		GPIO_LED_1->setDirection("out");
		GPIO_interlock_out->setDirection("out");
		GPIO_PSU_button_enable->setDirection("in");
	}
#endif

	/* If Linux OS + RPi HAT (ideally Raspberry Pi) -> enable interlock label and set correct string
	 * If Windows -> disable the interlock label */
#if defined(Q_OS_LINUX)
	ui->label_interlock_1->setVisible(config->get_support_HAT_B14_0835());
	ui->label_interlock_1->setEnabled(config->get_support_HAT_B14_0835());
	ui->label_interlock_1->setText("Power to modules is DISABLED.");
#elif defined(Q_OS_WINDOWS)
	ui->label_interlock_1->setVisible(false);
#endif


	/* Show or hide the various mode controls in settings menu */
	/* Luckily PWM triggering and Clock source have correct defaults, so disabling the buttons doesn't result in a misconfigured SG board. */
	ui->pushButton_PWM_triggering_free_running_1->setVisible(config->get_support_PWM_triggering());	//Firmware direction for this function is currently unclear. Disabled for the time being.
	ui->pushButton_PWM_triggering_free_running_1->setEnabled(config->get_support_PWM_triggering());
	ui->pushButton_PWM_triggering_master_1->setVisible(config->get_support_PWM_triggering());	//Firmware direction for this function is currently unclear. Disabled for the time being.
	ui->pushButton_PWM_triggering_master_1->setEnabled(config->get_support_PWM_triggering());
	ui->pushButton_PWM_triggering_slave_1->setVisible(config->get_support_PWM_triggering());	//Firmware direction for this function is currently unclear. Button generates an error, it has been disabled for the time being.
	ui->pushButton_PWM_triggering_slave_1->setEnabled(config->get_support_PWM_triggering());
	ui->PWM_triggering_label->setVisible(config->get_support_PWM_triggering());
	ui->PWM_triggering_label->setEnabled(config->get_support_PWM_triggering());

	coherency_supported = (config->get_support_coherency_standalone() ||
						   config->get_support_coherency_LVDS_master() ||
						   config->get_support_coherency_LVDS_slave_end() ||
						   config->get_support_coherency_LVDS_slave_inline() ||
						   config->get_support_coherency_AUX_master() ||
						   config->get_support_coherency_AUX_slave_end());

	ui->pushButton_clock_source_standalone_1->setVisible(config->get_support_coherency_standalone());
	ui->pushButton_clock_source_standalone_1->setEnabled(config->get_support_coherency_standalone());
	ui->pushButton_clock_source_master_1->setVisible(config->get_support_coherency_LVDS_master());
	ui->pushButton_clock_source_master_1->setEnabled(config->get_support_coherency_LVDS_master());
	ui->pushButton_clock_source_slave_end_1->setVisible(config->get_support_coherency_LVDS_slave_end());
	ui->pushButton_clock_source_slave_end_1->setEnabled(config->get_support_coherency_LVDS_slave_end());
	ui->pushButton_clock_source_slave_inline_1->setVisible(config->get_support_coherency_LVDS_slave_inline());
	ui->pushButton_clock_source_slave_inline_1->setEnabled(config->get_support_coherency_LVDS_slave_inline());
	ui->label_clock_source_1->setEnabled(coherency_supported);
	ui->label_clock_source_1->setVisible(coherency_supported);

	RCM_supported = (config->get_support_RCM_mode_USB_blind() ||
					 (config->get_support_RCM_mode_RS232_blind() && config->get_support_HAT_B14_0835()) ||
					 (config->get_support_RCM_mode_RS485_blind() && config->get_support_HAT_B14_0835()) ||
					 config->get_support_RCM_mode_TCP_blind());
	ui->RCM_label->setEnabled(RCM_supported);
	ui->RCM_label->setVisible(RCM_supported);
	ui->pushButton_remote_command_OFF_1->setVisible(RCM_supported);
	ui->pushButton_remote_command_OFF_1->setEnabled(RCM_supported);
	ui->pushButton_remote_command_USB_1->setVisible(config->get_support_RCM_mode_USB_blind());
	ui->pushButton_remote_command_USB_1->setEnabled(config->get_support_RCM_mode_USB_blind());
	ui->pushButton_remote_command_RS232_1->setVisible(config->get_support_RCM_mode_RS232_blind() && config->get_support_HAT_B14_0835());
	ui->pushButton_remote_command_RS232_1->setEnabled(config->get_support_RCM_mode_RS232_blind() && config->get_support_HAT_B14_0835());
	ui->pushButton_remote_command_RS485_1->setEnabled(config->get_support_RCM_mode_RS485_blind() && config->get_support_HAT_B14_0835());
	ui->pushButton_remote_command_RS485_1->setVisible(config->get_support_RCM_mode_RS485_blind() && config->get_support_HAT_B14_0835());
	ui->pushButton_remote_command_TCP_1->setVisible(config->get_support_RCM_mode_TCP_blind() );
	ui->pushButton_remote_command_TCP_1->setEnabled(config->get_support_RCM_mode_TCP_blind());

	ui->pushButton_power_control_normal_1->setVisible(config->get_support_power_control_modes());
	ui->pushButton_power_control_normal_1->setEnabled(config->get_support_power_control_modes());
	ui->pushButton_power_control_analog_1->setVisible(config->get_support_power_control_modes() && config->get_support_AIS_mode());
	ui->pushButton_power_control_analog_1->setEnabled(config->get_support_power_control_modes() && config->get_support_AIS_mode());
	ui->pushButton_power_control_feedforward_1->setVisible(config->get_support_power_control_modes() && config->get_support_feedforward_control());
	ui->pushButton_power_control_feedforward_1->setEnabled(config->get_support_power_control_modes() && config->get_support_feedforward_control());
	ui->pushButton_power_control_amplifier_1->setVisible(config->get_support_power_control_modes() && config->get_support_amplifier_mode());
	ui->pushButton_power_control_amplifier_1->setEnabled(config->get_support_power_control_modes() && config->get_support_amplifier_mode());
	ui->label_power_control_mode_1->setEnabled(config->get_support_power_control_modes());
	ui->label_power_control_mode_1->setVisible(config->get_support_power_control_modes());

	ui->pushButton_external_triggering_OFF_1->setVisible(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_OFF_1->setEnabled(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_ON_1->setVisible(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_ON_1->setEnabled(config->get_support_external_triggering_mode());
	ui->External_triggering_label->setVisible(config->get_support_external_triggering_mode());
	ui->External_triggering_label->setEnabled(config->get_support_external_triggering_mode());

	ui->menu_power_Button->setVisible(config->get_menu_power_enabled());		//enable or disable the power menu
	ui->menu_power_Button->setEnabled(config->get_menu_power_enabled());
	ui->menu_settings_Button->setVisible(config->get_menu_settings_enabled());	//enable or disable the settings menu
	ui->menu_settings_Button->setEnabled(config->get_menu_settings_enabled());
	ui->menu_sweep_Button->setVisible(config->get_menu_sweep_enabled());		//enable or disable the Sweep menu
	ui->menu_sweep_Button->setEnabled(config->get_menu_sweep_enabled());
	ui->menu_home_Button->setVisible(config->get_menu_home_enabled());			//enable or disable the home menu
	ui->menu_home_Button->setEnabled(config->get_menu_home_enabled());
	ui->menu_DLL_Button->setVisible(config->get_menu_DLL_enabled());			//enable or disable the DLL menu
	ui->menu_DLL_Button->setEnabled(config->get_menu_DLL_enabled());

	ui->pushButton_CW_enable_1->setVisible(config->get_controls_CW_enabled());		//enable or disable CW button
	ui->pushButton_CW_enable_1->setEnabled(config->get_controls_CW_enabled());
	ui->pushButton_PWM_enable_1->setVisible(config->get_controls_PWM_enabled());	//enable or disable PWM button
	ui->pushButton_PWM_enable_1->setEnabled(config->get_controls_PWM_enabled());
	ui->pushButton_DLL_enable_1->setVisible(config->get_controls_DLL_enabled());	//enable or disable DLL button
	ui->pushButton_DLL_enable_1->setEnabled(config->get_controls_DLL_enabled());

	/* Hide PSU buttons as necessary */
	ui->menu_PSU_Button->setVisible(config->get_menu_PSU_enabled());			//enable or disable the PSU menu
	ui->menu_PSU_Button->setEnabled(config->get_menu_PSU_enabled());
	ui->pushButton_PSU_enable_1->setVisible(config->get_support_PSU_controls_enable());
	ui->pushButton_PSU_voltage_1->setVisible(config->get_support_PSU_controls_voltage());
	ui->pushButton_PSU_current_limit_1->setVisible(config->get_support_PSU_controls_current_limit());
	ui->label_PSU_enable_1->setVisible(config->get_support_PSU_controls_enable());
	ui->label_PSU_voltage_1->setVisible(config->get_support_PSU_controls_voltage());
	ui->label_PSU_current_limit_1->setVisible(config->get_support_PSU_controls_current_limit());

	/* Disable PSU labels in the gridLayout based on PSU count */
	for (int i = ui->gridLayout_PSU_readings_2->columnCount() - 1; i >= 0; i--)	//columns / x-axis
	{
		for (int j = 0; j <= 4; j++)	//rows / y-axis
		{
			if (i >= config->get_PSU_count())
			{
				ui->gridLayout_PSU_readings_2->itemAtPosition(j,i)->widget()->setVisible(false);
			}
		}
	}

	/* PSU handling - prep QLists for value storage */
	PSU_count = config->get_PSU_count();
	for(int i = 0; i < PSU_count; i++)
	{
		PSU_enable.append(false);
		PSU_voltage.append(0);
		PSU_current.append(0);
		PSU_power.append(0);
	}

	/* Configure the channel number to be used for commands. */
	channel_select = config->get_channel_select();

	/* Setup serial communication to Signal Generator board; use first viable port provided by get_SGX_port(); */
	//
	// TODO:
	// GUI crashes 'index out of range' here if no SGx is connected and the list of ports comes up empty.
	//
	serial_Setup(SG_port, get_SGX_port().at(0).portName());	//Port for SGx connection.

	/* Initiate serial communication */
	if (SG_port.open(QIODevice::ReadWrite))
	{
		qDebug() << "Serial Setup -> Connection Established" << "\n";
		startup_sequence();
	}
	else
	{
		qDebug() << "\nSerial Setup -> Error opening port:" <<  SG_port.errorString() << "\n>Check if serial port isn't engaged" << "\n";
		QMessageBox message;
		message.critical(nullptr, "SERIAL CONNECTION ERROR", "Application could not connect to the signal generator board."
													   "\nSerial port error: " + SG_port.errorString());
		qDebug() << "The serial connection could not be established. Terminating program." << "\n";

		#ifndef DEBUG
		this->close();
		#endif
	}

	/* For Remote Command Mode prepare TCP Socket & Server. */
	tcpServer = new QTcpServer(this);
	tcpSocket = new QTcpSocket(this);

	/* ALL_capable is only valid after running through reset_restore() in the startup_sequence */
	ui->menu_ALL_Button->setVisible(config->get_menu_ALL_enabled());			//enable or disable the ALL menu
	ui->menu_ALL_Button->setEnabled(config->get_menu_ALL_enabled());
	ui->pushButton_ALL_enable_1->setVisible(config->get_controls_ALL_enabled());		//enable or disable the ALL button
	ui->pushButton_ALL_enable_1->setEnabled(config->get_controls_ALL_enabled());

	/* Initiate numpad */
	numpad = new Numpad(5,4, ui->numpadLabel_display, ui->numpadLabel_unit, ui->numpadButton_period, ui->numpadButton_sign);
	connect(numpad, &Numpad::value_confirmed_signal, this, &MainWindow::numpad_value_confirmed_handler);

	/* Connect Signals and Slots for polling and error handling */
	status_checker = new StatusCheck(&SG_port, channel_select, QCoreApplication::applicationDirPath() + "/messages.txt", config->get_console_output_enabled());
	connect(readings_timer, &QTimer::timeout, status_checker, &StatusCheck::poll_SG_error);

	if (config->get_read_power() == true)
	{
		connect(readings_timer, &QTimer::timeout, this, &MainWindow::get_PowerInOut);
	}

	connect(status_checker, &StatusCheck::SG_error_signal, this, &MainWindow::SG_error_handler);
	connect(status_checker, &StatusCheck::reset_detected_signal, this, &MainWindow::SG_reset_handler);
	connect(status_checker, &StatusCheck::RF_disabled_signal, this, &MainWindow::Suppress_RF_enable);

	ui->label_error_2->setText(status_checker->error_map.value("none"));

	/* optionally enable the temperature measurements */
	if (config->get_read_temperature() > 0 && PA_type != 0 && PA_type != 1)
	{
		connect(readings_timer, SIGNAL(timeout()), this, SLOT(get_Temperature()));
	}
	else
	{
		ui->label_temperature_1->setVisible(false);
		ui->label_temperature_2->setVisible(false);
		ui->label_temperature_3->setVisible(false);
	}

	/* Raspberry Pi HAT B14_0835 activities */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		/* Connect the LED timer to blink the LED if necessary */
		LED_timer = new QTimer(this);			//Timer for LED blinking
		connect(LED_timer, &QTimer::timeout, this, &MainWindow::LED_blink);

		/* Set GPIO 20 high. */
		GPIO_interlock_out->setValue("1");

		/* Start checking the enable state of system front panel button */
		connect(readings_timer, SIGNAL(timeout()), this, SLOT(handler_GPIO_PSU_button_enable()));

		GPIO_PSU_timer = new QTimer(this);		//Single-shot timer for delayed reaction to front panel button press.
		GPIO_PSU_timer->setSingleShot(true);
		connect(GPIO_PSU_timer, SIGNAL(timeout()), this, SLOT(handler_GPIO_PSU_timer()));
	}
#endif

	/* I sincerely wish I could just throw this out at this point.
	 * I don't know why I still haven't to be honest... */
	#ifdef ROTARY_ENCODER_ENABLED
	/* F&M Startup stuff */
	ui->menu_home_Button->setVisible(false);
	ui->menu_ALL_Button->setVisible(false);
	ui->menu_settings_Button->setVisible(false);
	ui->menu_help_Button->setVisible(false);
	ui->menu_sweep_Button->setVisible(false);
	ui->horizontalSpacer_2->changeSize(0,0,QSizePolicy::Fixed,QSizePolicy::Minimum);
	ui->horizontalSpacer_3->changeSize(0,0,QSizePolicy::Fixed,QSizePolicy::Minimum);
	ui->horizontalSpacer_4->changeSize(0,0,QSizePolicy::Fixed,QSizePolicy::Minimum);
	ui->horizontalSpacer_5->changeSize(0,0,QSizePolicy::Fixed,QSizePolicy::Minimum);

	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Rotary_Encoder_page));
	ui->pushButton_RF_enable_1->setChecked(false);
	ui->RF_enableButton_2->setChecked(false);

	 //making steps of 0.2, so going to 0 watt results in INFINITE dBm...
	 //therefor 0.2 is the starting value, and lowest value
	 power_watt_value = 0.2;


	RE_thread->moveToThread(thread1);
	connect(thread1, SIGNAL(started()), RE_thread, SLOT(process()));			//start the process, when thread is started
	connect(RE_thread, SIGNAL(error(QString)), this, SLOT(thread_ErrorHandler1(QString)));   //error handling
	connect(RE_thread, SIGNAL(knob_enter()), this, SLOT(thread_Knob_enterHandler()));
	connect(RE_thread, SIGNAL(knob_left()), this, SLOT(thread_Knob_leftHandler()));
	connect(RE_thread, SIGNAL(knob_right()), this, SLOT(thread_Knob_rightHandler()));
	connect(RE_thread, SIGNAL(manual_mode()), this, SLOT(thread_ManualHandler()));
	connect(RE_thread, SIGNAL(remote_command_mode()), this, SLOT(thread_TransparentHandler()));

	connect(RE_thread, SIGNAL(accelerate()), this, SLOT(thread_AccelerationHandler()));
	connect(RE_thread, SIGNAL(decelerate()), this, SLOT(thread_DecelerationHandler()));
	connect(this, SIGNAL(RE_threadSTOP()), RE_thread, SLOT(stop_process()));
	connect(RE_thread, SIGNAL(finished()), thread1, SLOT(quit()));			   //shut down thread when finished
	connect(RE_thread, SIGNAL(finished()), RE_thread, SLOT(deleteLater()));	  //prevent crashes due to thread not shutting down before it's deleted
	connect(thread1, SIGNAL(finished()), thread1, SLOT(deleteLater()));		  //prevent crashes due to thread not shutting down before it's deleted

	thread1->start();
	RE_menu_indicator();

	RE_update_timer = new QTimer(this);
	connect(RE_update_timer, SIGNAL(timeout()), this, SLOT(RE_update_values()));
	#endif

	/* set watt readings default */
	ui->pushButton_unit_S11_1->click();

	/* set Home menu default */
	ui->menu_home_Button->click();

	/* Load a stylesheet */
	QFile stylesheetfile(QCoreApplication::applicationDirPath() + "/theme.css");
	QString style;
	stylesheetfile.open(QFile::ReadOnly);
	style = stylesheetfile.readAll();
	stylesheetfile.close();

	/* Set stylesheet across the board */
	this->setStyleSheet(style);

	/* This is necessary so the logo image loads correctly regardless of the working path. Image needs to be in the same folder as the executable. */
	ui->logo_label->setStyleSheet("image: url(" + QCoreApplication::applicationDirPath() + "/logo.png);\nimage-position: top;\n");

	/* Thing that checks if unread data was left serial bus, ideally should never be triggering. */
	connect(this, SIGNAL(buffer_fillup_detected()), this, SLOT(buffer_fillup_handler()));

	RCM_mode_handler();
	/* Start the timer for periodic data readings.
	 * This timer MUST be disabled while RCM mode is running; SG's replies to commands will go to the remote client causing the data loss detection to kick in. */
	if (!(remote_command_mode > REMOTE_COMMAND_OFF))
	{
		if (config->get_continuous_readings_enabled() == true)
		{
			readings_timer->start(config->get_polling_rate_data());
		}
	}

	/* DEBUG build indication */
	#ifdef DEBUG
	show_warning("THIS IS A DEBUG BUILD!");
	show_notification("THIS IS A DEBUG BUILD!");
	ui->settings_plainTextEdit->appendPlainText(">\tTHIS IS A DEBUG BUILD!");
	#endif

	/* declare that startup phase has finished and program is ready to use. Used to prevent an infinite loop when dealing with RCM mode */
	bootup = false;
}

MainWindow::~MainWindow()
{
	/* Linux OS + HAT supported (ideally only Raspberry Pi)
	 * set GPIOs to sensible default values and set the direction to 'in', so that they are as close to inert as possible without unexporting. */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external->setValue("0");
		GPIO_control_SGx->setValue("1");
		GPIO_HAT_serial_mode->setValue("0");	//this is kind of pointless since it engages rs232 or rs485 on the HAT... but somehow rs232 feels more like a default state.
		GPIO_RS485_RW_flipper->setValue("0");
		GPIO_LED_1->setValue("0");

		GPIO_control_external->setDirection("in");
		GPIO_control_SGx->setDirection("in");
		GPIO_HAT_serial_mode->setDirection("in");
		GPIO_RS485_RW_flipper->setDirection("in");
		GPIO_LED_1->setDirection("in");
	}
#endif

	if(SG_port.isOpen())
	{
		//serial_parser(serial_setRF_enable(serial, channel_select, false));
		//necessary in case of alt+f4 while RCM is enabled.
		if (remote_command_mode > REMOTE_COMMAND_OFF)
		{
			ui->pushButton_remote_command_OFF_1->click();	//enable control again.
		}

		ECG_parser();

		while(RF_enabled != false)
		{
			serial_setRF_enable(SG_port,channel_select,false);
			ECG_parser();
		}
	}

	#ifdef ROTARY_ENCODER_ENABLED
	emit RE_threadSTOP();
	#endif

	//serial_parser(serial_setIdle_enable(serial, channel_select, true));
	//thread1->quit();	//maybe unnecessary?
	SG_port.close();
	RCM_port.close();
	delete ui;
}

/**********************************************************************************************************************************************************************************
 * UTILITY FUNCTIONS
 * *******************************************************************************************************************************************************************************/
//
// TODO:
// It'd be cool if we could have a generic Serialport get function that could be used by providing filter parameters?
// Instead of using the overly specific auto-detect functions below.
//
/*
 * SGx board identifiers:
 * Vendor:	0x1fc9	= 8137
 * Product:	0x0083	= 131
 *
 * USB-TTL / USB-NMC cable identifiers
 * Vendor:	0x403	= 1027
 * Product:	0x6001	= 24577
 */
/* Pick a serialport for the SGx: If config file contains an output_port use that; otherwise auto-detect. */
QList<QSerialPortInfo> MainWindow::get_SGX_port()
{
	QList<QSerialPortInfo> infolist;
	if (config->get_SGX_port(1) != "")
	{
		QSerialPortInfo info(config->get_SGX_port(1));
		if (!info.isNull())
		{
			infolist += info;
			qDebug() << info.portName();
		}
		if (infolist.isEmpty())
		{
			QMessageBox message;
			message.critical(nullptr,	"COULD NOT DETECT DEVICE",
								"No device detected at specified port: " + config->get_SGX_port(1) +
								"\nSerial connection cannot be established.\n"
								"\n1. Ensure signal generator board is connected."
								"\n2. Ensure the correct port name is specified at SGX_port_1 in config.txt."
								"\n3. Try again.");
		}
	}
	else
	{
		for (const QSerialPortInfo& info : QSerialPortInfo::availablePorts())
		{
			if ((info.vendorIdentifier() == 8137 && info.productIdentifier() == 131))
			{
				infolist += info;
				qDebug() << info.portName();
			}
		}

		if (infolist.isEmpty())
		{
			QMessageBox message;
			message.critical(nullptr,	"COULD NOT AUTO-DETECT SIGNAL GENERATOR BOARD",
								"No signal generator board auto-detected at any port."
								"\nSerial connection cannot be established.\n"
								"\n1. Ensure signal generator board is connected."
								"\n2. Try again."
								"\n3. If problem persist try configuring a port name for SGX_port_1 in config.txt");
		}
	}

	if (infolist.size() > 1)
	{
		qDebug() << "Multiple signal generator boards found";
	}

	if (infolist.isEmpty())
	{
		this->close();
	}

	return infolist;
}


/* Pick a serialport for Remote Command Mode: If config file contains an output_port use that; otherwise auto-detect a non-SGX port. */
QList<QSerialPortInfo> MainWindow::get_RCM_port()
{
	/* if an FTDI NMC cable is available use that; otherwise pick whatever is not an SGx board */
	QList<QSerialPortInfo> infolist;
	for (const QSerialPortInfo& info : QSerialPortInfo::availablePorts())
	{
		if (info.vendorIdentifier() == 1027 && info.productIdentifier() == 24577)
		{
			infolist += info;
			qDebug() << info.portName();
		}
	}
	if (infolist.isEmpty())
	{
		for (const QSerialPortInfo& info : QSerialPortInfo::availablePorts())
		{
			if (info.vendorIdentifier() != 8137 && info.productIdentifier() != 131)
			{
				infolist += info;
				qDebug() << info.portName();
			}
		}
	}

	if (infolist.size() > 1)
	{
		qDebug() << "Multiple RCM cables found";
	}

	if (infolist.isEmpty())
	{
		qDebug() << "No RCM cables found";
	}

	return infolist;
}

bool MainWindow::port_open(QSerialPort *port)
{
	/* Retry reconnecting a few times (Windows is a tad slow to assign COM ports if there are a lot of devices...) */
	for (int attempt = 0, retries = 100, total_time = 10000;	attempt < retries; attempt++)
	{
		//If doing this for the SGX port, first detect available ports.
		if(port == &SG_port)
		{
			if(SG_port.error() != QSerialPort::NoError)
			{
				qDebug() << SG_port.error() << SG_port.errorString();
				SG_port.close();
				QList<QSerialPortInfo> temp_list = serial_get_SGX_port_quiet();
				if (!temp_list.isEmpty())
				{
					SG_port.setPortName(temp_list.at(0).portName());
					qDebug() << SG_port.portName();
				}
			}
		}

		// Total time willing to spend trying to (re)connect paced by amount of attempts to be made.
		QThread::msleep(total_time / retries);
		qDebug() << "Connection attempt: " << attempt+1 << "/" << retries;
		if (!port->open(QIODevice::ReadWrite))
		{
			if (attempt == retries-1)
			{
				QMessageBox message;
				message.critical(nullptr,"Communication dropped", "Failed to reconnect to SGx board following a reset\n Software will shutdown");
				this->close();
				return false;
			}
		}
		else
		{
			qDebug() << "Connection successful." << "\n";
			break;
		}
	}
	return true;
}

void MainWindow::startup_sequence()
{
	/* Reset board at startup, clear anything that may have been left over and is unaccounted for. */
	if (config->get_reset_on_startup() == true)
	{
		safe_reset();
	}
	else
	{
		ui->settings_plainTextEdit->appendPlainText(">\tWarning: Reset is disabled. Settings remain unchanged.\n\tGUI may not accurately reflect the state of the SG Board!");
	}
	reset_restore();
}

/* Safe_reset performs a clean reset action.
 * Data polling enable / disable is handled automatically.
 * After initiating the reset, serial coms are reset automatically, PA type is set and the dedicated shutdown check is done */
void MainWindow::safe_reset()
{
	/* Stop the constant readings */
	readings_timer->stop();

	/* Before doing anything check if the firmware version is compatible with this software
	 * Some old firmware versions won't even send an $RST,1,OK and won't properly display the version message */
	VER_parser();
	version_control();

	/* Do a reset; if succesful do stuff, else DIE */
	if (serial_parser(serial_reset(SG_port,channel_select)) == true)
	{
		/* Following a reset, USB communication drops and needs to be reinitiated
		 * close the communication and wait a bit before opening coms again */
		SG_port.close();

		/* Retry reconnecting a few times (Windows is a tad slow to assign COM ports if there are a lot of devices...) */
		port_open(&SG_port);

		/* Set the appropriate PA type (unless -1, in which case just let it be).
		 * Clear out errors the PA type may generate initially */
		if(config->get_target_PA_type() >= 0)
		{
			/* Set the new PA type and */
			serial_parser(serial_set_PA_type(SG_port,channel_select, PA_type));
			serial_parser(serial_clearError(SG_port,channel_select));
		}

		/* Pre-emptively check the state of the SGx.
		 * If the only thing going on is a reset and an external shutdown, send an error clear to avoid unnecessary messages for the user
		 * Otherwise situation is unexpected and a pop-up message is warranted.*/
		// TODO: When firmware supports it, allow error clearing only the related error bits
		ST_parser();
		serial_clearError(SG_port,channel_select);

		/* Ensure RF is disabled at startup / following a reset */
		if(RF_enabled != ui->pushButton_RF_enable_1->isChecked())
		{
			ui->pushButton_RF_enable_1->click();
			qDebug() << "It's concerning if you're seeing this; RF output was left enabled following a reset procedure";
		}
	}
	else
	{
		this->close();
	}

	if (config->get_continuous_readings_enabled() == true && bootup == false)
	{
		readings_timer->start(config->get_polling_rate_data());
	}
}

/* This function is intended to be called following a system reset, whether intentional or unintentional.
 * It configures the GUI into the same state it's in at normal startup.*/
void MainWindow::reset_restore()
{
	qDebug() << "Restoring from Reset";

	/* Set the starting values as defined in the config file */
	sync_to_config();

	/* Determine the type of SG board connected */
	handle_SG_type();

	/* Disable RF power for safety */
	serial_parser(serial_setRF_enable(SG_port, channel_select, false));

	/* Check if the software version is compatible with firmware */
	VER_parser();
	version_control();

	/* Set Chain mode, PWM triggering, External Triggering, Power Control modes, Set ALL / DLL / PWM enable states, Configure PSU */
	handle_config_targets();

	/* Ensure values from config file are actually sensible */
	startup_check_minmax();

	//
	// TODO:
	// There is a PA set both in reset_restore and and in safe_reset. Only one should stay. Also it's not 100% clear whether the config target should be used or the local PA_type variable);
	// previous PATG is causing this command to get sent needlessly...
	//
	PATG_parser();
	if(config->get_target_PA_type() >= 0)
	{
		if(config->get_target_PA_type() != PA_type)
		{
			serial_parser(serial_set_PA_type(SG_port,channel_select, config->get_target_PA_type()));
		}
	}

	/* Ensure DVGA ($FWDS and $RFLS) are set correctly at startup, as well as that $AIS has the right pre-configuration.
	 * Configuring the DVGAs correctly is very important for systems that use the SGx board's internal detectors! */
	DVGA_enable();

	if (power_control_mode == POWER_CONTROL_NORMAL)
	{
		serial_parser(serial_setPower_dbm(SG_port, channel_select, power_dbm_value));											//Set the power output value
	}

	serial_parser(serial_setFrequency(SG_port, channel_select, freq_value*0.000001));											//Set the output frequency value
	serial_parser(serial_setPhase(SG_port, channel_select, phase_value));														//set the output phase value
	serial_parser(serial_setPWM_freq (SG_port, channel_select, PWM_freq_value, PWM_correction_factor, PWM_delay));				//set the PWM frequency value

	if (ext_trig_mode == 0)		//External trigger requires DC 0%, so don't overwrite it.
	{
		serial_parser(serial_setPWM_enable(SG_port, channel_select, PWM_enabled, PWM_duty_cycle_value));
	}

	if(config->get_menu_DLL_enabled() == true)
	{
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value,
										 DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));	//Set the DLL settings
	}

	if(config->get_menu_ALL_enabled() == true)
	{
		serial_parser(serial_setALL_settings(SG_port, channel_select, ALL_freqLimit_DOWN_value, ALL_freqLimit_UP_value, ALL_threshold_value));	//Set the ALL settings
		if(ALL_enabled && config->get_support_DVGA())
		{
			serial_parser(serial_set_DVGA_forward(SG_port, channel_select, DVGA_amplifier_forward_enabled, DVGA_attenuation_forward));		//set the VGA forward amplifier and attentuation
			serial_parser(serial_set_DVGA_reflected(SG_port, channel_select, DVGA_amplifier_reflected_enabled, DVGA_attenuation_reflected));	//set the VGA reflected amplifier and attentuation
		}
		else
		{
			if (config->get_support_DVGA() == true)
			{
				//set the DVGA's to safe default values.
				serial_parser(serial_set_DVGA_forward(SG_port, channel_select, config->get_default_DVGA_amplifier_forward_enabled(), config->get_default_DVGA_attenuation_forward()));		//set the VGA forward amplifier and attentuation
				serial_parser(serial_set_DVGA_reflected(SG_port, channel_select, config->get_default_DVGA_amplifier_reflected_enabled(), config->get_default_DVGA_attenuation_reflected()));	//set the VGA reflected amplifier and attentuation
			}
		}
	}
	else
	{
		if (config->get_support_DVGA() == true)
		{
			//set the DVGA's to safe default values.
			serial_parser(serial_set_DVGA_forward(SG_port, channel_select, config->get_default_DVGA_amplifier_forward_enabled(), config->get_default_DVGA_attenuation_forward()));		//set the VGA forward amplifier and attentuation
			serial_parser(serial_set_DVGA_reflected(SG_port, channel_select, config->get_default_DVGA_amplifier_reflected_enabled(), config->get_default_DVGA_attenuation_reflected()));		//set the VGA reflected amplifier and attentuation
		}
	}

	if (config->get_target_RF_enable() == true)
	{
		ui->pushButton_RF_enable_1->setChecked(config->get_target_RF_enable());
		on_pushButton_RF_enable_1_clicked();
		show_notification("CAUTION: RF is auto-enabled");
	}

	run_all_gets();
	update_labels();	//update the button labels/text
}

void MainWindow::sync_to_config()
{
	/* Defaults */
	PA_type				= config->get_target_PA_type();
	ext_trig_mode		= config->get_ext_triggering_mode();
//	remote_command_mode	= config->get_remote_command_mode();
//	clock_source		= config->get_chain_mode();
//	PWM_trigger_mode	= config->get_PWM_triggering_mode();

	/* CW mode */
	power_watt_value	= config->get_power_watt_value();
	power_dbm_value		= config->get_power_dbm_value();
	freq_value			= config->get_frequency_value();	//TODO: fix name
	phase_value			= config->get_phase_value();

	/* PWM mode */
	PWM_enabled				= config->get_PWM_enabled();
	PWM_duty_cycle_value	= config->get_PWM_duty_cycle_value();
	PWM_freq_value			= config->get_PWM_frequency_value();	//TODO: fix name
	PWM_master_port			= config->get_PWM_master_port();
	PWM_master_pin			= config->get_PWM_master_pin();
	PWM_slave_port			= config->get_PWM_slave_port();
	PWM_slave_pin			= config->get_PWM_slave_pin();

	/* ALL mode */
//	ALL_enabled							= config->get_ALL_enabled();
	ALL_freqLimit_UP_value				= config->get_ALL_freqLimit_UP_value();
	ALL_freqLimit_DOWN_value			= config->get_ALL_freqLimit_DOWN_value();
	ALL_threshold_value					= config->get_ALL_threshold_value();
	DVGA_attenuation_forward			= config->get_DVGA_forward_attenuation();
	DVGA_attenuation_reflected			= config->get_DVGA_reflected_attenuation();
	DVGA_amplifier_forward_enabled		= config->get_DVGA_forward_amplifier_enabled();
	DVGA_amplifier_reflected_enabled	= config->get_DVGA_reflected_amplifier_enabled();

	/* DLL mode */
//	DLL_enabled					= config->get_DLL_enabled();
	DLL_freqLimit_UP_value		= config->get_DLL_freqLimit_UP();
	DLL_freqLimit_DOWN_value	= config->get_DLL_freqLimit_DOWN();
	DLL_freq_start_value		= config->get_DLL_frequency_start();
	DLL_freq_step_value			= config->get_DLL_frequency_step();
	DLL_threshold_value			= config->get_DLL_threshold_db();
	DLL_main_delay_value		= config->get_DLL_delay_ms();

	/* Sweep */
	SWP_start_freq				= config->get_SWP_start_freq();
	SWP_stop_freq				= config->get_SWP_stop_freq();
	SWP_step_freq				= config->get_SWP_step_freq();
	SWP_power_dbm				= config->get_SWP_power_dbm();
	SWP_power_watt				= config->get_SWP_power_watt();
	SWP_maximum_measure_points	= config->get_SWP_maximum_measure_points(); //TODO: fix name

	/* Power Control Mode */
	power_control_mode			= config->get_power_control_mode();

	/* Analog Input Mode */
	AIS_attenuation_value		= config->get_AIS_attenuation_value();
}


void MainWindow::handle_SG_type()
{
	QString get_string = serial_getIdentity(SG_port,channel_select);
	if (get_string.contains("$IDN,") && get_string.contains("\r\n"))
	{
		get_string.remove("\r\n");
	}
	else
	{
		QMessageBox message;
		message.critical(nullptr, "Hardware not recognized!", "The signal generator board is not recognized.\nTerminating program.");
		this->close();
	}

	QStringList get_list = get_string.split(",");

	SG_manufacturer = get_list.at(2);
	SG_board_model = get_list.at(3);

	if (get_list.count() >= 5)
	{
		SG_serial_number = get_list.at(4);
	}
	else
	{
		SG_serial_number = "-";
	}

#ifdef DEBUG
	ui->SG_board_identity_label_2->setText(SG_board_model);
#else
	ui->SG_board_identity_label_1->setVisible(false);
	ui->SG_board_identity_label_2->setVisible(false);
#endif
}

void MainWindow::run_all_gets()
{
	ST_parser();
	VER_parser();
	ECG_parser();
	FCG_parser();
	PCG_parser();
	if (config->get_read_temperature() > 0 && PA_type != 0 && PA_type != 1)
	{
		PTG_parser();
	}
	PWRG_parser();
	PWRDG_parser();
	if (config->get_menu_DLL_enabled() == true)
	{
		DLL_DLCG_parser();
		DLL_DLEG_parser();
	}
//	PWM_DCG_parser();
	PWM_DCG_frequency_parser();
	PWM_DCG_duty_cycle_parser();
	PWM_DCG_PWM_trig_parser();
	PATG_parser();
	CSG_parser();
	//GCG_parser();	//Don't run GCG, will overwrite the setting from config...

	if (config->get_menu_ALL_enabled() == true)
	{
		RFLG_parser();
		FWDG_parser();
		ALL_ALCG_parser();
		ALL_ALEG_parser();
	}
}

#warning misnomer -> updates button text.
void MainWindow::update_labels()
{
	//HOME
	ui->pushButton_power_watt_1->setText(zeroChopper(QString::number(power_watt_value, 'f', 1)));
	ui->pushButton_power_dbm_1->setText(zeroChopper(QString::number(power_dbm_value, 'f',1)));
	ui->pushButton_frequency_1->setText(zeroChopper(QString::number(freq_value * 0.000001, 'f', 3)));
	ui->pushButton_phase_1->setText(zeroChopper(QString::number(phase_value, 'f', 1)));
	ui->pushButton_PWM_frequency_1->setText(zeroChopper(QString::number(PWM_freq_value, 'f', 2)));
	ui->pushButton_PWM_duty_cycle_1->setText(zeroChopper(QString::number(PWM_duty_cycle_value, 'f', 1)));
	ui->pushButton_DLL_step_frequency_1->setText(zeroChopper(QString::number(DLL_freq_step_value, 'f', 2)));
	ui->pushButton_DLL_threshold_1->setText(zeroChopper(QString::number(DLL_threshold_value, 'f', 2)));
	ui->pushButton_ALL_threshold_1->setText(zeroChopper(QString::number(ALL_threshold_value, 'f', 2)));

	/* Only update SSG_power and power_dbm_standalone if it's the last thing that was used, because its GET value is not accurate after any other method of power modulation has been used.
	 * i.e. PWRDS / PWRSGDS correlates to specific MCS/GCS values, but not the other way around.
	 * If another method has been used, change it to dash and leave it like that until SSG_power/power_dbm_standalone has been changed directly again. */
	if (target_parameter_button == ui->pushButton_power_SGx_dbm_1)
	{
		ui->pushButton_power_SGx_dbm_1->setText(zeroChopper(QString::number(sgx_power_value,'f',1)));
		ui->pushButton_power_dbm_standalone_1->setText("-");
	}
	else if( target_parameter_button == ui->pushButton_power_dbm_standalone_1)
	{
		ui->pushButton_power_dbm_standalone_1->setText(zeroChopper(QString::number(power_dbm_standalone_value,'f',1)));
		ui->pushButton_power_SGx_dbm_1->setText("-");
	}
	else if (
				target_parameter_button == ui->pushButton_VGA_attenuation_1
			 || target_parameter_button == ui->pushButton_IQMod_magnitude_1
			 || target_parameter_button == ui->pushButton_power_dbm_1
			 || target_parameter_button == ui->pushButton_power_dbm_2
			 || target_parameter_button == ui->pushButton_power_watt_1
			 || target_parameter_button == ui->pushButton_power_watt_2
			 || target_parameter_button == ui->pushButton_SWP_power_dbm_1
			 || target_parameter_button == ui->pushButton_SWP_power_watt_1)
	{
		ui->pushButton_power_SGx_dbm_1->setText("-");
		ui->pushButton_power_dbm_standalone_1->setText("-");
	}

	ui->pushButton_VGA_attenuation_1->setText(zeroChopper(QString::number(VGA_attenuation_value,'f',1)));
	ui->pushButton_IQMod_magnitude_1->setText(zeroChopper(QString::number(magnitude_value,'f',1)));

	//ALL
	ui->pushButton_ALL_threshold_2->setText(zeroChopper(QString::number(ALL_threshold_value, 'f', 2)));
	ui->pushButton_ALL_frequency_limit_upper_1->setText(zeroChopper(QString::number(ALL_freqLimit_UP_value, 'f', 2)));
	ui->pushButton_ALL_frequency_limit_lower_1->setText(zeroChopper(QString::number(ALL_freqLimit_DOWN_value, 'f', 2)));

	if(ALL_enabled && config->get_support_DVGA())
	{
		ui->pushButton_DVGA_attenuation_forward_1->setText(zeroChopper(QString::number(DVGA_attenuation_forward, 'f', 2)));
		ui->pushButton_DVGA_attenuation_reflected_1->setText(zeroChopper(QString::number(DVGA_attenuation_reflected, 'f', 2)));
	}

	//DLL
	ui->pushButton_power_watt_2->setText(zeroChopper(QString::number(power_watt_value, 'f', 1)));
	ui->pushButton_power_dbm_2->setText(zeroChopper(QString::number(power_dbm_value, 'f',1)));
	ui->pushButton_DLL_step_frequency_2->setText(zeroChopper(QString::number(DLL_freq_step_value, 'f',2)));
	ui->pushButton_DLL_start_frequency_1->setText(zeroChopper(QString::number(DLL_freq_start_value, 'f',2)));
	ui->pushButton_DLL_threshold_2->setText(zeroChopper(QString::number(DLL_threshold_value, 'f', 2)));
	ui->pushButton_DLL_main_delay_1->setText(zeroChopper(QString::number(DLL_main_delay_value,'f',2)));
	ui->pushButton_DLL_frequency_limit_upper_1->setText(zeroChopper(QString::number(DLL_freqLimit_UP_value, 'f', 2)));
	ui->pushButton_DLL_frequency_limit_lower_1->setText(zeroChopper(QString::number(DLL_freqLimit_DOWN_value, 'f', 2)));

	//SWEEP
	ui->pushButton_SWP_start_frequency_1->setText(zeroChopper(QString::number(SWP_start_freq * 0.000001, 'f',3)));
	ui->pushButton_SWP_stop_frequency_1->setText(zeroChopper(QString::number(SWP_stop_freq * 0.000001, 'f', 3)));
	ui->pushButton_SWP_step_frequency_1->setText(zeroChopper(QString::number(SWP_step_freq * 0.000001, 'f', 3)));
	ui->pushButton_SWP_power_dbm_1->setText(zeroChopper(QString::number(SWP_power_dbm, 'f', 1)));
	ui->pushButton_SWP_power_watt_1->setText(zeroChopper(QString::number(SWP_power_watt, 'f', 1)));

	//HELP/About information
	ui->firmware_version_label_2->setText(SG_firmware_version);
}

void MainWindow::handle_config_targets()
{
	/* Set Chain mode, PWM triggering, Power Control mode,  External Triggering */
	Coherency_mode_handler();
	PWM_triggering_mode_handler();
	Power_control_mode_handler();
	External_triggering_handler();


	/* Set ALL / DLL / PWM */
	/* ALL enable */
	if(config->get_menu_ALL_enabled() == true)
	{
		if (ALL_enabled != config->get_ALL_enabled())
		{
			ui->pushButton_ALL_enable_1->setChecked(config->get_ALL_enabled());
			on_pushButton_ALL_enable_1_clicked();
		}
	}
	show_ALL_settings(ALL_enabled);

	/* DLL enable */
	if (config->get_menu_DLL_enabled() == true)
	{
		if (DLL_enabled != config->get_DLL_enabled())
		{
			ui->pushButton_DLL_enable_1->setChecked(config->get_DLL_enabled());
			on_pushButton_DLL_enable_1_clicked();
		}
	}
	show_DLL_settings(DLL_enabled);

	/* PWM (enable) configure */
	if (config->get_controls_PWM_enabled() == true)
	{
		if (PWM_enabled != config->get_PWM_enabled())
		{
			ui->pushButton_PWM_enable_1->setChecked(config->get_PWM_enabled());
			on_pushButton_PWM_enable_1_clicked();
		}
	}
	show_PWM_settings(PWM_enabled);


	/* Set PSU configurations */
	handler_PSU_reset_to_default();

}

/* slightly differs from ok_check_minmax because with startup config there is no perspective of changes occuring.
 * All values are defined simultaneously, therefore there is no way to correct 'properly'.
 * startup_check_minmax does not make use of *target_parameter */
void MainWindow::startup_check_minmax()
{
	//CW PWR WATT
	if (power_watt_value > config->get_power_watt_max()){
		power_watt_value = zeroChopper(QString::number(config->get_power_watt_max(),'f',6)).toDouble();
	}
	else if (power_watt_value < config->get_power_watt_min()){
		power_watt_value = zeroChopper(QString::number(config->get_power_watt_min(),'f',6)).toDouble();
	}

	//CW PWR DBM
	if (power_dbm_value > convert_watt_to_dbm(config->get_power_watt_max())){
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_max()),'f',6)).toDouble();
	}
	else if(power_dbm_value < convert_watt_to_dbm(config->get_power_watt_min())){
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_min()),'f',6)).toDouble();
	}

	// SSG Power dBm for standalone mode with PA type 1.
	if (power_dbm_standalone_value > power_dbm_standalone_max)
	{
		power_dbm_standalone_value = power_dbm_standalone_max;
	}
	else if (power_dbm_standalone_value < power_dbm_standalone_min)
	{
		power_dbm_standalone_value = power_dbm_standalone_min;
	}

	//CW FREQUENCY MHZ
	if (freq_value > config->get_frequency_max()){
		freq_value = config->get_frequency_max();
	}
	else if (freq_value < config->get_frequency_min()){
		freq_value = config->get_frequency_min();
	}

	//CW PHASE
	if (phase_value > 360.0){
		phase_value = fmod(phase_value, 360.0);	//modulo with decimals...
	}
	else if (phase_value < 0){
		phase_value = 0;
	}

	//PWM FREQUENCY HZ
	if (PWM_freq_value < config->get_PWM_frequency_min()){
		PWM_freq_value = config->get_PWM_frequency_min();
	}
	if (PWM_freq_value > config->get_PWM_frequency_max()){
		PWM_freq_value = config->get_PWM_frequency_max();
	}

	//PWM DUTY CYCLE
	if (config->get_power_control_mode() ==  0) //Only use limit if We're in normal control mode (Auto-gain enabled)
	{
		PWM_duty_cycle_min = PWM_freq_value * config->get_PWM_minimum_pulse_length_ns() / 10000000;
		double leftover = fmod(PWM_duty_cycle_min, 1.0);

		if (leftover > 0 && leftover < 0.5){
			PWM_duty_cycle_min = round(PWM_duty_cycle_min + 1);
		}
		else
		{
			PWM_duty_cycle_min = round(PWM_duty_cycle_min);
		}
	}
	else
	{
		PWM_duty_cycle_min = 1;
	}

	if (PWM_duty_cycle_value > 99){
		PWM_duty_cycle_value = 99;
	}
	else if (PWM_duty_cycle_value < PWM_duty_cycle_min)
	{
		show_notification("Duty cycle must exceed " + QString::number(PWM_duty_cycle_min) + "% at " + QString::number(PWM_freq_value) + "Hz PWM frequency.");
		PWM_duty_cycle_value = PWM_duty_cycle_min;
	}

	//DLL UPPER & LOWER FREQUENCY
	if (DLL_freqLimit_UP_value > config->get_frequency_max() * 0.000001){
		DLL_freqLimit_UP_value = config->get_frequency_max() * 0.000001;
	}
	else if (DLL_freqLimit_UP_value <= config->get_frequency_min() * 0.000001){
		DLL_freqLimit_UP_value = config->get_frequency_min() * 0.000001 + 1;
	}

	if (DLL_freqLimit_DOWN_value < config->get_frequency_min() * 0.000001){
		DLL_freqLimit_DOWN_value = config->get_frequency_min() * 0.000001;
	}
	else if (DLL_freqLimit_DOWN_value >= config->get_frequency_max() * 0.000001){
		DLL_freqLimit_DOWN_value = config->get_frequency_max() * 0.000001 - 1;
	}

	if (DLL_freqLimit_UP_value <= DLL_freqLimit_DOWN_value){
		DLL_freqLimit_UP_value = DLL_freqLimit_DOWN_value + 1;
	}

	if (DLL_freqLimit_DOWN_value >= DLL_freqLimit_UP_value){
		DLL_freqLimit_DOWN_value = DLL_freqLimit_UP_value - 1;	//Avoid issues if upper freq = 2500 MHz
	}

	//DLL START FREQUENCY
	if (DLL_freq_start_value > DLL_freqLimit_UP_value){
		DLL_freq_start_value = DLL_freqLimit_UP_value;
	}
	else if (DLL_freq_start_value < DLL_freqLimit_DOWN_value){
		DLL_freq_start_value = DLL_freqLimit_DOWN_value;
	}

	//DLL STEP FREQUENCY
	if (DLL_freq_step_value > (DLL_freqLimit_UP_value - DLL_freqLimit_DOWN_value)){
		DLL_freq_step_value = (DLL_freqLimit_UP_value - DLL_freqLimit_DOWN_value);
	}
	else if (DLL_freq_step_value < config->get_DLL_freqLimit_step()){
		DLL_freq_step_value = config->get_DLL_freqLimit_step();
	}

	//DLL THRESHOLD
	if (DLL_threshold_value < 0){
		DLL_threshold_value = 0;
	}
	else if (DLL_threshold_value > DLL_threshold_max){
		DLL_threshold_value = DLL_threshold_max;
	}

	//DLL MAIN DELAY
	if (DLL_main_delay_value < DLL_main_delay_min){
		DLL_main_delay_value = DLL_main_delay_min;
	}
	else if (DLL_main_delay_value > DLL_main_delay_max){
		DLL_main_delay_value = DLL_main_delay_max;
	}

	//ALL UPPER & LOWER FREQUENCY
	if (ALL_freqLimit_UP_value > config->get_ALL_frequency_max()){
		ALL_freqLimit_UP_value = config->get_ALL_frequency_max();
	}
	else if (ALL_freqLimit_UP_value <= config->get_ALL_frequency_min()){
		ALL_freqLimit_UP_value = config->get_ALL_frequency_min() + 0.01;		//Avoid issues if lower freq = 2400 MHz
	}

	if (ALL_freqLimit_DOWN_value < config->get_ALL_frequency_min()){
		ALL_freqLimit_DOWN_value = config->get_ALL_frequency_min();
	}
	else if (ALL_freqLimit_DOWN_value >= config->get_ALL_frequency_max()){
		ALL_freqLimit_DOWN_value = config->get_ALL_frequency_max() - 0.01;		//Avoid issues if lower freq = 2400 MHz
	}

	if (ALL_freqLimit_UP_value <= ALL_freqLimit_DOWN_value){
		ALL_freqLimit_UP_value = ALL_freqLimit_DOWN_value + 0.01;	//Avoid issues if upper freq = 2500 MHz
	}

	if (ALL_freqLimit_DOWN_value >= ALL_freqLimit_UP_value){
		ALL_freqLimit_DOWN_value = ALL_freqLimit_UP_value - 0.01;	//Avoid issues if upper freq = 2500 MHz
	}

	//ALL THRESHOLD
	if (ALL_threshold_value < 0){
		ALL_threshold_value = 0;
	}
	else if (ALL_threshold_value > 3.3)	{
		ALL_threshold_value = 3.3;
	}

	//VGA ATTENUATION FORWARD
	if (DVGA_attenuation_forward < 0){
		DVGA_attenuation_forward = 0;
	}
	else if (DVGA_attenuation_forward > 31.5){
		DVGA_attenuation_forward = 31.5;
	}

	//VGA ATTENUATION REFLECTED
	if (DVGA_attenuation_reflected < 0){
		DVGA_attenuation_reflected = 0;
	}
	else if (DVGA_attenuation_reflected > 31.5){
		DVGA_attenuation_reflected = 31.5;
	}

	//SWEEP START & STOP FREQUENCY
	if (SWP_stop_freq > config->get_frequency_max()){
		SWP_stop_freq = config->get_frequency_max();
	}
	else if (SWP_stop_freq <= config->get_frequency_min()){
		SWP_stop_freq = config->get_frequency_min() + 1000000;
	}

	if (SWP_start_freq < config->get_frequency_min()){
		SWP_start_freq = config->get_frequency_min();
	}
	else if (SWP_start_freq >= config->get_frequency_max()){
		SWP_start_freq = config->get_frequency_max() - 1000000;
	}

	if (SWP_stop_freq <= SWP_start_freq){
		SWP_stop_freq = SWP_start_freq + 1000000;
	}

	if (SWP_start_freq >= SWP_stop_freq){
		SWP_start_freq = SWP_stop_freq - 1000000;	//Avoid issues if upper freq = 2500 MHz
	}

	//SWEEP STEP FREQUENCY
	if (SWP_step_freq > 100000000.0){
		SWP_step_freq = 100000000.0;
	}
	else if ((((SWP_stop_freq - SWP_start_freq) / SWP_step_freq) + 1) > SWP_maximum_measure_points)
	{
		//These calculations are in Hz, instead of MHz, so everything is multiplied by 1.000.000
		SWP_step_freq = QString::number(((SWP_stop_freq - SWP_start_freq) / (SWP_maximum_measure_points - 1))/1000000, 'f', 2).toDouble() * 1000000;
		show_notification("Cannot sweep more than " + QString::number(SWP_maximum_measure_points) + " points in the frequency band.");
	}

	//SWEEP POWER WATT
	if (SWP_power_watt > config->get_power_watt_max()){
		SWP_power_watt = zeroChopper(QString::number(config->get_power_watt_max(),'f',6)).toDouble();
	}
	else if (SWP_power_watt < config->get_power_watt_min()){
		SWP_power_watt = zeroChopper(QString::number(config->get_power_watt_min(),'f',6)).toDouble();
	}

	//SWEEP POWER DBM
	if (SWP_power_dbm > convert_watt_to_dbm(config->get_power_watt_max())){
		SWP_power_dbm = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_max()),'f',6)).toDouble();
	}
	else if(SWP_power_dbm < convert_watt_to_dbm(config->get_power_watt_min())){
		SWP_power_dbm = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_min()),'f',6)).toDouble();
	}

	if (power_control_mode == POWER_CONTROL_AMPLIFIER)
	{
		if (VGA_attenuation_value > config->get_AM_attenuation_max_db())
		{
			VGA_attenuation_value = config->get_AM_attenuation_max_db();
		}
		else if (VGA_attenuation_value < config->get_AM_attenuation_min_db())
		{
			VGA_attenuation_value = config->get_AM_attenuation_min_db();
		}
	}
}



/*
 * Handle LED lighting if the B14_0835 HAT is used
 * Mode 0 - RF OFF – LED OFF
 * Mode 1 - RF ON – LED ON (solid)
 * Mode 2 - ALARM - LED Blink (500ms ON -> 500ms OFF)
 * Mode 3 - Blind RCM - LED Blink (1000ms ON -> 1000ms OFF)
 */
#if defined(Q_OS_LINUX)
void MainWindow::LED_handler()
{
	if (config->get_support_HAT_B14_0835())
	{
		int LED_mode = 0;

		if (remote_command_mode != REMOTE_COMMAND_OFF)
		{
			LED_mode = 3;
		}
		else if (ui->label_error_2->text() != status_checker->error_map.value("none"))
		{
			LED_mode = 2;
		}
		else if (RF_enabled == true)
		{
			LED_mode = 1;
		}
		else
		{
			LED_mode = 0;
		}

		if (LED_mode != last_LED_mode)
		{
			switch (LED_mode)
			{
				case 0:
					GPIO_LED_1->setValue("0");
					LED_timer->stop();
					break;
				case 1:
					GPIO_LED_1->setValue("1");
					LED_timer->stop();
					break;
				case 2:
					GPIO_LED_1->setValue("1");  //flip ON for the error state
					LED_timer->start(500);		//Fast blink for errors
					break;
				case 3:
					GPIO_LED_1->setValue("1");  //flip ON for the error state
					LED_timer->start(1000);		//Slow blink for RCM
					break;
				default:
					GPIO_LED_1->setValue("0");
					LED_timer->stop();
					break;
			}
		}

		last_LED_mode = LED_mode;
	}
}

/* Invert the GPIO_LED value every time this function is called */
void MainWindow::LED_blink()
{
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_LED_1->setValue(QString::number(!(bool)GPIO_LED_1->readValue()));
	}
}

#endif


/* Raspberry Pi HAT B14_0835 + Linux feature only. */
/* This is a solution / workaround to the problem that the GUI/SGx cannot communicate with the PSU if the front panel switch isn't enabled, including potentially at startup of the device */
/* Read GPIO to see if the enable switch on the Front Panel of a system is enabled.
 * Display the 'interlock' notification */
#if defined(Q_OS_LINUX)
void MainWindow::handler_GPIO_PSU_timer()
{
	/* Check the button state again, to ensure the PSU is really enabled and it wasn't just a false positive. */

	bool button_enable = read_GPIO_PSU_button_enable();

	if (button_enable == false)
	{
		serial_parser(serial_setRF_splitter_channels_connect(SG_port, channel_select, 0x00));
//		GPIO_interlock_out->setValue("0");
		serial_parser(serial_setPSU_enable(SG_port, channel_select, button_enable));

	}
	else if (button_enable == true)
	{
		/* Use config settings to determine if some form of PSU control is enabled. Could be either a config default setting that the GUI sends or user controls.
		 * If that is the case, whenever the front panel switch is flipped, the GUI should update the setting using the latest value that was provided. */
		if ((config->get_PSU_voltage() > 0) || (config->get_support_PSU_controls_voltage() == true))
		{
//			serial_parser(serial_setPSU_voltage_setpoint(SG_port, channel_select, PSU_voltage_setpoint));
			#warning temporary workaround
			serial_parser(serial_setPSU_voltage_setpoint(SG_port, channel_select, config->get_PSU_voltage()));
		}
		if ((config->get_PSU_current_limit() > 0) || (config->get_support_PSU_controls_current_limit() == true))
		{
//			serial_parser(serial_setPSU_current_limit(SG_port, channel_select, PSU_current_limit));
			#warning temporary workaround
			serial_parser(serial_setPSU_voltage_setpoint(SG_port, channel_select, config->get_PSU_voltage()));
		}
	
		/* If the RPI hat is used, the front panel button dictates the enable State of the PSU */
		if (config->get_support_HAT_B14_0835() == true)		//TODO: this if statement is probably unnecessary?
		{
//			GPIO_interlock_out->setValue("1");
			serial_parser(serial_setPSU_enable(SG_port, channel_select, button_enable));
			serial_parser(serial_setRF_splitter_channels_connect(SG_port, channel_select, 0x0f));
		}
//		else
//		{
//			// For the time being the (software) PSU enable should remain unaffected by the front panel button.
//			if ((config->get_PSU_enabled() >= 0) || (config->get_support_PSU_controls_enable() == true))
//			{
//				serial_parser(serial_setPSU_enable(SG_port, channel_select, PSU_enable_combined));
//			}
//		}
	}
}


void MainWindow::handler_GPIO_PSU_button_enable()
{
	if (config->get_support_HAT_B14_0835())
	{
		bool GPIO_PSU_button_enable_state = read_GPIO_PSU_button_enable();

		/* Front panel button state changed */
		if (last_GPIO_PSU_button_enable_state != GPIO_PSU_button_enable_state)
		{
//			GPIO_PSU_timer->start(1000);
			handler_GPIO_PSU_timer();
		}


		/* Toggle interlock label visibility when there is a change.
		 * Remark: 'isEnabled()' is used, because 'isVisible()' returns false when looking at pages that don't have the side panel on screen, such as the Settings menu */
		if (ui->label_interlock_1->isEnabled() == GPIO_PSU_button_enable_state)	
		{
			ui->label_interlock_1->setEnabled(!GPIO_PSU_button_enable_state);
			ui->label_interlock_1->setVisible(!GPIO_PSU_button_enable_state);
			qDebug() << "PSU switch enable state: " << GPIO_PSU_button_enable_state;
		}

		last_GPIO_PSU_button_enable_state = GPIO_PSU_button_enable_state;
	}
}

bool MainWindow::read_GPIO_PSU_button_enable()
{
	if (config->get_support_HAT_B14_0835())
	{
		bool GPIO_PSU_button_enable_state = false;

		if (GPIO_PSU_button_enable->readValue() > 0)		//If the front panel button is pressed.
		{
			GPIO_PSU_button_enable_state = true;
		}
		return GPIO_PSU_button_enable_state;
	}
}
#endif

/**********************************************************************************************************************************************************************************
 * Show mode settings
 * *******************************************************************************************************************************************************************************/
/* Show the PWM buttons in the home menu */
void MainWindow::show_PWM_settings(bool input)
{
	ui->PWM_freq_label->setVisible(input);
	ui->PWM_dc_label->setVisible(input);
	ui->pushButton_PWM_frequency_1->setVisible(input);
	ui->pushButton_PWM_duty_cycle_1->setVisible(input);
}

/* Show the DLL buttons in the home menu */
void MainWindow::show_DLL_settings(bool input)
{
	ui->DLL_stepFreq_label->setVisible(input);
	ui->DLL_threshold_label->setVisible(input);
	ui->pushButton_DLL_step_frequency_1->setVisible(input);
	ui->pushButton_DLL_threshold_1->setVisible(input);
}

/* Show the ALL buttons in the home menu */
void MainWindow::show_ALL_settings(bool input)
{
	ui->ALL_threshold_label->setVisible(input);
	ui->label_3->setVisible(input);
	ui->pushButton_ALL_threshold_1->setVisible(input);
}

/* Show the HOME menu power controls for autogain, enable the DLL menu power controls for autogain, enable the SWEEP menu Sweep Power controls */
void MainWindow::show_autogain_controls(bool enable)
{
	/* enable/disable power controls */
	ui->pushButton_power_dbm_1->setEnabled(enable);
	ui->pwr_dbm_label->setEnabled(enable);
	ui->pushButton_power_watt_1->setEnabled(enable);
	ui->pwr_watt_label->setEnabled(enable);
	ui->pushButton_power_dbm_2->setEnabled(enable);
	ui->DLL_power_dbm_label->setEnabled(enable);
	ui->pushButton_power_watt_2->setEnabled(enable);
	ui->DLL_power_watt_label->setEnabled(enable);

	ui->pushButton_power_dbm_1->setVisible(enable);
	ui->pwr_dbm_label->setVisible(enable);
	ui->pushButton_power_watt_1->setVisible(enable);
	ui->pwr_watt_label->setVisible(enable);

	ui->SWP_power_dbm_label->setEnabled(enable);
	ui->SWP_power_watt_label->setEnabled(enable);
	ui->pushButton_SWP_power_dbm_1->setEnabled(enable);
	ui->pushButton_SWP_power_watt_1->setEnabled(enable);
}

/* Disable any controls that utilize the IQ Modulator. Also hide the RF Enable button. It is not indicative of anything in this mode. */
void MainWindow::show_Amplifier_mode_controls(bool enable)
{
	ui->pushButton_RF_enable_1->setEnabled(!enable);
	ui->label_power_SGx_dbm_1->setEnabled(!enable);
	ui->pushButton_power_SGx_dbm_1->setEnabled(!enable);
	ui->pushButton_IQMod_magnitude_1->setEnabled(!enable);
	ui->label_IQMod_magnitude_1->setEnabled(!enable);
//	ui->pushButton_frequency_1->setEnabled(!enable);		//Leave Frequency button available so that calibration / power measurements can be adjusted to the freqency of the external oscillator source.
//	ui->freq_label->setEnabled(!enable);
	ui->pushButton_phase_1->setEnabled(!enable);
	ui->pushButton_phase_1->setVisible(!enable);
	ui->phase_label->setEnabled(!enable);
	ui->phase_label->setVisible(!enable);
	ui->pushButton_RF_enable_1->setVisible(!enable);

	if (ext_trig_mode == 0)
	{
		ui->menu_sweep_Button->setEnabled(!enable);
	}
	else
	{
		ui->menu_sweep_Button->setEnabled(false);
	}

	ui->menu_DLL_Button->setEnabled(!enable);
	ui->menu_ALL_Button->setEnabled(!enable);
	ui->pushButton_DLL_enable_1->setEnabled(!enable);
	ui->pushButton_ALL_enable_1->setEnabled(!enable);

//	ui->pushButton_external_triggering_OFF_1->click();
	ui->External_triggering_label->setEnabled(!enable);
	ui->pushButton_external_triggering_ON_1->setEnabled(!enable);
	ui->pushButton_external_triggering_OFF_1->setEnabled(!enable);

	ui->pushButton_PWM_enable_1->setEnabled(!enable && (ext_trig_mode == 0));
}

/* Show the standalone (PA type 1) SSG power button ($PWRDS) in the home menu */
void MainWindow::show_power_dbm_standalone_controls(bool enable)
{
	ui->label_power_dbm_standalone_1->setVisible(enable);
	ui->pushButton_power_dbm_standalone_1->setVisible(enable);
}

/* Show the SSG power button ($PWRSGDS) in the home menu */
void MainWindow::show_SGx_Power_controls(bool enable)
{
	ui->label_power_SGx_dbm_1->setVisible(enable);
	ui->pushButton_power_SGx_dbm_1->setVisible(enable);
}

/* Show the GCS button in the home menu */
void MainWindow::show_GCS_controls(bool enable)
{
	ui->label_VGA_attenuation_1->setVisible(enable);
	ui->pushButton_VGA_attenuation_1->setVisible(enable);

	if (config->get_support_HAT_B14_0835())
	{
		ui->label_VGA_attenuation_1->setEnabled(!enable);
		ui->pushButton_VGA_attenuation_1->setEnabled(!enable);
	}
}

/* Show the MCS button in the home menu */
void MainWindow::show_MCS_controls(bool enable)
{
	ui->label_IQMod_magnitude_1->setVisible(enable);
	ui->pushButton_IQMod_magnitude_1->setVisible(enable);
}


/**********************************************************************************************************************************************************************************
 * Polling stuff
 * *******************************************************************************************************************************************************************************/

void MainWindow::get_PowerInOut()
{
	/* Read FWD and Reflected powers in dBm */
	serial_readPower_dbm(SG_port, channel_select, PowerFRW_Value, PowerRFL_Value, config->get_console_output_enabled());

#ifdef ROTARY_ENCODER_ENABLED
	ui->reflected_power_label_2->setText(zeroChopper(QString::number(convert_dbm_to_watt(PowerRFL_Value), 'f',2)));
	ui->forward_power_label_2->setText(zeroChopper(QString::number(convert_dbm_to_watt(PowerFRW_Value), 'f',2)));
	ui->S11_label_2->setText(zeroChopper(QString::number((convert_dbm_to_watt(PowerRFL_Value)/convert_dbm_to_watt(PowerFRW_Value))*100, 'f',2)));
	ui->temperature_label_2->setText(zeroChopper(QString::number(Temperature_Value, 'f',2)));
#endif

	//
	// TODO:
	// Some kind of indication in the labels that PWM compensation is happening would be useful
	//

	/* If PWM is enabled and Duty Cycle compensation is enabled -> Subtract dB's from power measurements */
	if (config->get_duty_cycle_compensation_enabled() == true && PWM_enabled == true)
	{
		/* Duty Cycle works as a divisor -> Yields negative dB values */
		PowerFRW_Value += convert_percent_to_dB(PWM_duty_cycle_value);
		PowerRFL_Value += convert_percent_to_dB(PWM_duty_cycle_value);
	}

	/* Update the values in the labels */
	if (ui->pushButton_unit_power_forward_1->isChecked() == true)
	{
		ui->label_power_forward->setText(QString::number(convert_dbm_to_watt(PowerFRW_Value), 'f',1));
	}
	else
	{
		ui->label_power_forward->setText(QString::number(PowerFRW_Value, 'f',1));
	}

	if (ui->pushButton_unit_power_reflected_1->isChecked() == true)
	{
		ui->label_power_reflected->setText(QString::number(convert_dbm_to_watt(PowerRFL_Value), 'f',1));
	}
	else
	{
		ui->label_power_reflected->setText(QString::number(PowerRFL_Value, 'f',1));
	}

	if(PowerFRW_Value >= 30)	//1.0W
	{
		if(ui->pushButton_unit_S11_1->isChecked()==true)
		{
			/* Perhaps theoretically a division by zero is possible here, but the minimum power value that can be set is 0.1W / 20dBm anyway...
			 * Since we get our power values in dBm and convert to watt, there's no such thing as 0 watt anyway... therefor in practice divide by zero doesn't occur. */
			ui->label_s11_value->setText(QString::number((convert_dbm_to_watt(PowerRFL_Value) / convert_dbm_to_watt(PowerFRW_Value)) *100, 'f',0));
		}
		else
		{
			ui->label_s11_value->setText(QString::number((PowerRFL_Value-PowerFRW_Value), 'f',1));
		}
	}
	else
	{
		ui->label_s11_value->setText("-");
	}
}

void MainWindow::get_Temperature()
{
	PTG_parser(config->get_console_output_enabled());

	Temperature_average = 0;

	if (Temperature_array.count() > config->get_read_temperature())
	{
		Temperature_array.removeFirst();
	}

	Temperature_array.append(Temperature_Value);

	for (int i = 0; i < Temperature_array.count(); i++)
	{
		Temperature_average += Temperature_array.at(i);
	}

	Temperature_average /= Temperature_array.count();
	ui->label_temperature_2->setText(QString::number(Temperature_average, 'f',1));
}

void MainWindow::get_IQmodVGA()
{
	MCG_parser(false);
	GCG_parser(false);
	ui->pushButton_VGA_attenuation_1->setText(zeroChopper(QString::number(VGA_attenuation_value,'f',1)));
	ui->pushButton_IQMod_magnitude_1->setText(zeroChopper(QString::number(magnitude_value,'f',1)));
}

/**********************************************************************************************************************************************************************************
 * Error Handling
 * *******************************************************************************************************************************************************************************/
void MainWindow::SG_error_handler(quint64 *error)
{
	QList<QString> error_descriptions = status_checker->translate_SG_error(*error);

	if (*error > 0)
	{
		/* display an appropriate error message. */
		QString error_message;
		for (int i = 0; i < error_descriptions.size(); i++)
		{
			error_message += error_descriptions[i] + "\n";
		}
		error_message.chop(1);
		show_warning(error_message);
		ui->label_error_2->setText(QString("0x%1").arg(*error,0,16));
	}
	else
	{
		/* If no error, omit the error code and show the value mapped to 'none' */
		close_warning();
		ui->label_error_2->setText(status_checker->error_map.value("none"));
	}


	/* Update LED status */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		LED_handler();
	}
#endif
}

void MainWindow::SG_reset_handler()
{
	reset_restore();
	show_notification("A reset has occured");
	//reset restore I guess?
}

void MainWindow::Suppress_RF_enable()
{
	if (RF_enabled == true)
	{
		ui->pushButton_RF_enable_1->setChecked(false);
		on_pushButton_RF_enable_1_clicked();
	}
	//if RF power is enabled, disable RF_power
}


/**********************************************************************************************************************************************************************************
 * POWER MENU BUTTONS
 * *******************************************************************************************************************************************************************************/

void MainWindow::on_pushButton_shutdown_controller_1_clicked()
{
	power_menu_action = 1;
	ui->power_menu_confirm_label->setText("Are you sure you want to shut down?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void MainWindow::on_pushButton_reboot_controller_1_clicked()
{
	power_menu_action = 2;
	ui->power_menu_confirm_label->setText("Are you sure you want to reboot?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void MainWindow::on_pushButton_exit_program_1_clicked()
{
	power_menu_action = 3;
	ui->power_menu_confirm_label->setText("Are you sure you want to exit the program?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

//ToDo: I wish I could just combine this with the destructor and get rid of the extra code...
//Maybe I could trigger a delayed shutdown and then activate the destructor.
void MainWindow::on_Exit_Yes_button_clicked()
{

	if (remote_command_mode > REMOTE_COMMAND_OFF){
		ui->pushButton_remote_command_OFF_1->click();	//enable control again.
	}
	QProcess process;
	ECG_parser();
	while(RF_enabled != false)
	{
		serial_setRF_enable(SG_port,channel_select,false);
		ECG_parser();	//necessary
	}
	switch(power_menu_action)
	{
		case 1:
			process.startDetached("sudo poweroff -f");
			this->close();
			break;
		case 2:
			process.startDetached("sudo reboot -f");
			this->close();
			break;
		case 3:
			this->close();
			break;
		default:
			break;
	}
}

void MainWindow::on_Exit_No_button_clicked()
{
	ui->stackedWidget_3->setCurrentIndex(last_page_index);
}

/**********************************************************************************************************************************************************************************
 * Parameter buttons
 * *******************************************************************************************************************************************************************************/

// HOME
void MainWindow::on_pushButton_power_dbm_1_clicked(){
	numpad_parameter_select(ui->pushButton_power_dbm_1, &power_dbm_value, "dBm", 1, 1);
}
void MainWindow::on_pushButton_power_watt_1_clicked(){
	numpad_parameter_select(ui->pushButton_power_watt_1, &power_watt_value, "watt", 1, 1);
}
void MainWindow::on_pushButton_frequency_1_clicked(){
	numpad_parameter_select(ui->pushButton_frequency_1, &freq_value, "MHz", 3, 0.000001);
}
void MainWindow::on_pushButton_phase_1_clicked(){
	numpad_parameter_select(ui->pushButton_phase_1, &phase_value, "°", 0, 1, false);
}
void MainWindow::on_pushButton_PWM_frequency_1_clicked(){
	numpad_parameter_select(ui->pushButton_PWM_frequency_1, &PWM_freq_value, "Hz", 0, 1, false);
}
void MainWindow::on_pushButton_PWM_duty_cycle_1_clicked(){
	numpad_parameter_select(ui->pushButton_PWM_duty_cycle_1, &PWM_duty_cycle_value, "%", 0, 1, false);
}
void MainWindow::on_pushButton_DLL_step_frequency_1_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_step_frequency_1, &DLL_freq_step_value, "MHz");
}
void MainWindow::on_pushButton_DLL_threshold_1_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_threshold_1, &DLL_threshold_value, "dB");
}
void MainWindow::on_pushButton_ALL_threshold_1_clicked(){
	numpad_parameter_select(ui->pushButton_ALL_threshold_1, &ALL_threshold_value, "V");
}

// HOME FEEDFWD / MANUAL POWER CONTROLS
void MainWindow::on_pushButton_power_dbm_standalone_1_clicked(){
	numpad_parameter_select(ui->pushButton_power_dbm_standalone_1, &power_dbm_standalone_value, "dBm", 1, 1, true, true);
}
void MainWindow::on_pushButton_power_SGx_dbm_1_clicked(){
	numpad_parameter_select(ui->pushButton_power_SGx_dbm_1, &sgx_power_value, "dBm", 1, 1, true, true);
}
void MainWindow::on_pushButton_VGA_attenuation_1_clicked(){
	numpad_parameter_select(ui->pushButton_VGA_attenuation_1, &VGA_attenuation_value, "dB");
}
void MainWindow::on_pushButton_IQMod_magnitude_1_clicked(){
	numpad_parameter_select(ui->pushButton_IQMod_magnitude_1, &magnitude_value, "%");
}

// ALL
void MainWindow::on_pushButton_ALL_frequency_limit_upper_1_clicked(){
	numpad_parameter_select(ui->pushButton_ALL_frequency_limit_upper_1, &ALL_freqLimit_UP_value, "V");
}
void MainWindow::on_pushButton_ALL_frequency_limit_lower_1_clicked(){
	numpad_parameter_select(ui->pushButton_ALL_frequency_limit_lower_1, &ALL_freqLimit_DOWN_value, "V");
}
void MainWindow::on_pushButton_ALL_threshold_2_clicked(){
	numpad_parameter_select(ui->pushButton_ALL_threshold_2, &ALL_threshold_value, "V");
}
void MainWindow::on_pushButton_DVGA_attenuation_forward_1_clicked(){
	numpad_parameter_select(ui->pushButton_DVGA_attenuation_forward_1, &DVGA_attenuation_forward, "dB");
}
void MainWindow::on_pushButton_DVGA_attenuation_reflected_1_clicked(){
	numpad_parameter_select(ui->pushButton_DVGA_attenuation_reflected_1, &DVGA_attenuation_reflected, "dB");
}

void MainWindow::on_pushButton_ALL_reference_open_clicked()
{
	if (RF_enabled == true){
		ui->pushButton_RF_enable_1->click();
	}
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->ALL_reference_page));
}

void MainWindow::on_pushButton_DVGA_amplifier_forward_1_clicked()
{
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	if(ui->pushButton_DVGA_amplifier_forward_1->isChecked() == true){
		ui->pushButton_DVGA_amplifier_forward_1->setText("ON");
		DVGA_amplifier_forward_enabled = true;
	}
	else{
		ui->pushButton_DVGA_amplifier_forward_1->setText("OFF");
		DVGA_amplifier_forward_enabled = false;
	}
	serial_parser(serial_set_DVGA_forward(SG_port, channel_select, DVGA_amplifier_forward_enabled, DVGA_attenuation_forward));
}

void MainWindow::on_pushButton_DVGA_amplifier_reflected_1_clicked()
{
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	if(ui->pushButton_DVGA_amplifier_reflected_1->isChecked() == true){
		ui->pushButton_DVGA_amplifier_reflected_1->setText("ON");
		DVGA_amplifier_reflected_enabled = true;
	}
	else{
		ui->pushButton_DVGA_amplifier_reflected_1->setText("OFF");
		ui->pushButton_DVGA_amplifier_reflected_1->setChecked(false);
		DVGA_amplifier_reflected_enabled = false;
	}
	serial_parser(serial_set_DVGA_reflected(SG_port, channel_select, DVGA_amplifier_reflected_enabled, DVGA_attenuation_reflected));
}

//The button on the ALL reference page that returns you to the main ALL page
void MainWindow::on_pushButton_ALL_reference_back_clicked()
{
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
}

// SWEEP
void MainWindow::on_pushButton_SWP_start_frequency_1_clicked(){
	numpad_parameter_select(ui->pushButton_SWP_start_frequency_1, &SWP_start_freq, "MHz", 3, 0.000001);
}
void MainWindow::on_pushButton_SWP_stop_frequency_1_clicked(){
	numpad_parameter_select(ui->pushButton_SWP_stop_frequency_1, &SWP_stop_freq, "MHz", 3, 0.000001);
}
void MainWindow::on_pushButton_SWP_step_frequency_1_clicked(){
	numpad_parameter_select(ui->pushButton_SWP_step_frequency_1, &SWP_step_freq, "MHz", 3, 0.000001);
}
void MainWindow::on_pushButton_SWP_power_dbm_1_clicked(){
	numpad_parameter_select(ui->pushButton_SWP_power_dbm_1, &SWP_power_dbm, "dBm", 1);
}
void MainWindow::on_pushButton_SWP_power_watt_1_clicked(){
	numpad_parameter_select(ui->pushButton_SWP_power_watt_1, &SWP_power_watt, "watt", 1);
}

// DLL
void MainWindow::on_pushButton_DLL_frequency_limit_lower_1_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_frequency_limit_lower_1, &DLL_freqLimit_DOWN_value, "MHz", 3);
}
void MainWindow::on_pushButton_DLL_frequency_limit_upper_1_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_frequency_limit_upper_1, &DLL_freqLimit_UP_value, "MHz", 3);
}
void MainWindow::on_pushButton_DLL_start_frequency_1_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_start_frequency_1, &DLL_freq_start_value, "MHz", 3);
}
void MainWindow::on_pushButton_DLL_step_frequency_2_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_step_frequency_2, &DLL_freq_step_value, "MHz", 3);
}
void MainWindow::on_pushButton_DLL_threshold_2_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_threshold_2, &DLL_threshold_value, "dB");
}
void MainWindow::on_pushButton_DLL_main_delay_1_clicked(){
	numpad_parameter_select(ui->pushButton_DLL_main_delay_1, &DLL_main_delay_value, "ms", 0, 1, false);
}
void MainWindow::on_pushButton_power_dbm_2_clicked(){
	numpad_parameter_select(ui->pushButton_power_dbm_2, &power_dbm_value, "dBm", 1);
}
void MainWindow::on_pushButton_power_watt_2_clicked(){
	numpad_parameter_select(ui->pushButton_power_watt_2, &power_watt_value, "watt", 1);
}

//PSU
void MainWindow::on_pushButton_PSU_voltage_1_clicked()
{
	numpad_parameter_select(ui->pushButton_PSU_voltage_1, &PSU_voltage_setpoint, "V", 1, 1, false);
}

void MainWindow::on_pushButton_PSU_current_limit_1_clicked()
{
	numpad_parameter_select(ui->pushButton_PSU_current_limit_1, &PSU_current_limit, "A", 1, 1, false);
}

void MainWindow::on_pushButton_PSU_enable_1_clicked(bool checked)
{
	if (config->get_support_HAT_B14_0835())
	{
		if (checked == false)
		{
			serial_parser(serial_setRF_splitter_channels_connect(SG_port, channel_select, 0x00));
		}
	}

	bool success = serial_parser(serial_setPSU_enable(SG_port, channel_select, checked));

	if (success)
	{
		handler_PSU_enable_combined_get(checked);
	}

	if (config->get_support_HAT_B14_0835())
	{
		if (checked == true)
		{
			serial_parser(serial_setRF_splitter_channels_connect(SG_port, channel_select, 0x0f));
		}
	}
}

/**********************************************************************************************************************************************************************************
 * NUMPAD BUTTONS
 * *******************************************************************************************************************************************************************************/

void MainWindow::on_numpadButton_0_clicked(){
	numpad->press_0();
}
void MainWindow::on_numpadButton_1_clicked(){
	numpad->press_1();
}
void MainWindow::on_numpadButton_2_clicked(){
	numpad->press_2();
}
void MainWindow::on_numpadButton_3_clicked(){
	numpad->press_3();
}
void MainWindow::on_numpadButton_4_clicked(){
	numpad->press_4();
}
void MainWindow::on_numpadButton_5_clicked(){
	numpad->press_5();
}
void MainWindow::on_numpadButton_6_clicked(){
	numpad->press_6();
}
void MainWindow::on_numpadButton_7_clicked(){
	numpad->press_7();
}
void MainWindow::on_numpadButton_8_clicked(){
	numpad->press_8();
}
void MainWindow::on_numpadButton_9_clicked(){
	numpad->press_9();
}
void MainWindow::on_numpadButton_period_clicked(){
	numpad->press_period();
}
void MainWindow::on_numpadButton_backspace_clicked(){
	numpad->press_delete();
}
void MainWindow::on_numpadButton_clear_all_clicked(){
	numpad->press_delete_all();
}
void MainWindow::on_numpadButton_arrow_left_clicked(){
	numpad->press_indicator_left();
}
void MainWindow::on_numpadButton_arrow_right_clicked(){
	numpad->press_indicator_right();
}
void MainWindow::on_numpadButton_plus_clicked(){
	numpad->press_plus();
}
void MainWindow::on_numpadButton_minus_clicked(){
	numpad->press_minus();
}
void MainWindow::on_numpadButton_sign_clicked(){
	numpad->press_sign();
}
void MainWindow::on_numpadButton_ok_clicked(){
	numpad->press_confirm_input();
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
}

/**********************************************************************************************************************************************************************************
 * FUNCTIONS FOR NUMPAD
 * *******************************************************************************************************************************************************************************/
void MainWindow::numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision, double scale,  bool period_enable, bool sign_enable)
{
	target_parameter_button = button;	//still used to close the numpad when pressing the DLL/ALL/PWM/CW buttons.
	numpad->setOutputButton(button);
	numpad->load_value(parameter, unit, period_enable, sign_enable, scale);

	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->numpad_page));
	scale_multiplier = scale;
	this->precision = precision;
}

void MainWindow::numpad_value_confirmed_handler(QPushButton *button, double *num)
{
	ok_check_minmax(button);
	QString str = zeroChopper(QString::number(*num * scale_multiplier,'f', precision));
	button->setText(str);

	//Home Page parameters
	if (button == ui->pushButton_frequency_1){
		serial_parser(serial_setFrequency(SG_port, channel_select, freq_value * scale_multiplier));

		/* PA type 1 has frequency compensation, so fetch MCS and GCS values after setting the frequency. */
		if (PA_type == 1)
		{
			MCG_parser();
			GCG_parser();
		}
	}
	if (button == ui->pushButton_power_dbm_1){
		power_watt_value = zeroChopper(QString::number(convert_dbm_to_watt(power_dbm_value),'f',1)).toDouble();  //Yes, I'm converting a double to an string and back, just so I can limit the digits... No this is definitely not smart... Yes... It does work. The loss in precision is so laughably small it doesn't make any sense to care about it for dBm's and watts.
		serial_parser(serial_setPower_dbm(SG_port, channel_select, power_dbm_value));
	}
	if (button == ui->pushButton_power_watt_1){
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(power_watt_value),'f',6)).toDouble();
		serial_parser(serial_setPower_watt(SG_port, channel_select, power_watt_value));
	}
	if (button == ui->pushButton_phase_1){
		serial_parser(serial_setPhase(SG_port, channel_select, phase_value));
	}
	if (button == ui->pushButton_PWM_frequency_1){
		serial_parser(serial_setPWM_freq(SG_port, channel_select, PWM_freq_value, PWM_correction_factor, PWM_delay));	//Minmax automatically changes the minimum PWM Duty Cycle value, which changes when the PWM Frequency is adjusted.
		serial_parser(serial_setPWM_DC(SG_port, channel_select, PWM_duty_cycle_value));
		show_notification("Minimum duty cycle value is " + QString::number(PWM_duty_cycle_min) + "% at this PWM frequency.");
	}
	if (button == ui->pushButton_PWM_duty_cycle_1){
		serial_parser(serial_setPWM_DC(SG_port, channel_select, PWM_duty_cycle_value));
	}
	if (button == ui->pushButton_DLL_step_frequency_1){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_DLL_threshold_1){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_ALL_threshold_1){
		serial_parser(serial_setALL_settings(SG_port, channel_select, ALL_freqLimit_DOWN_value, ALL_freqLimit_UP_value, ALL_threshold_value));
	}

	//Feedforward / Manual power controls
	/* Fetch MCS and GCS values after setting the feed forward power. */
	if (button == ui->pushButton_power_dbm_standalone_1){
		serial_parser(serial_setPower_dbm(SG_port, channel_select, power_dbm_standalone_value));
		MCG_parser();
		GCG_parser();
	}
	if (button == ui->pushButton_power_SGx_dbm_1){
		serial_parser(serial_setSGxPower_dbm(SG_port, channel_select, sgx_power_value));
		MCG_parser();
		GCG_parser();
	}
	if (button == ui->pushButton_VGA_attenuation_1){
		serial_parser(serial_setVGA_attenuation(SG_port, channel_select, VGA_attenuation_value));
	}
	if (button == ui->pushButton_IQMod_magnitude_1){
		serial_parser(serial_setMagnitude(SG_port, channel_select, magnitude_value));
	}

	//ALL Page parameters
	if (button == ui->pushButton_ALL_frequency_limit_upper_1){
		serial_parser(serial_setALL_settings(SG_port, channel_select, ALL_freqLimit_DOWN_value, ALL_freqLimit_UP_value, ALL_threshold_value));
	}
	if (button == ui->pushButton_ALL_frequency_limit_lower_1){
		serial_parser(serial_setALL_settings(SG_port, channel_select, ALL_freqLimit_DOWN_value, ALL_freqLimit_UP_value, ALL_threshold_value));
	}
	if (button == ui->pushButton_ALL_threshold_2){
		serial_parser(serial_setALL_settings(SG_port, channel_select, ALL_freqLimit_DOWN_value, ALL_freqLimit_UP_value, ALL_threshold_value));
	}
	if (button == ui->pushButton_DVGA_attenuation_forward_1){
		serial_parser(serial_set_DVGA_forward(SG_port, channel_select, DVGA_amplifier_forward_enabled, DVGA_attenuation_forward));
	}
	if (button == ui->pushButton_DVGA_attenuation_reflected_1){
		serial_parser(serial_set_DVGA_reflected(SG_port, channel_select, DVGA_amplifier_reflected_enabled, DVGA_attenuation_reflected));
	}

	//DLL Page parameters
	if (button == ui->pushButton_DLL_frequency_limit_upper_1){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_DLL_frequency_limit_lower_1){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_DLL_start_frequency_1){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_DLL_step_frequency_2){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_DLL_threshold_2){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_DLL_main_delay_1){
		serial_parser(serial_setDLL_settings(SG_port, channel_select, DLL_freqLimit_DOWN_value, DLL_freqLimit_UP_value, DLL_freq_start_value,DLL_freq_step_value, DLL_threshold_value, DLL_main_delay_value));
	}
	if (button == ui->pushButton_power_dbm_2){
		power_watt_value = zeroChopper(QString::number(convert_dbm_to_watt(power_dbm_value),'f',1)).toDouble();  //Yes, I'm converting a double to an string and back, just so I can limit the digits... No this is definitely not smart... Yes... It does work. The loss in precision is so laughably small it doesn't make any sense to care about it for dBm's and watts.
		serial_parser(serial_setPower_dbm(SG_port, channel_select, power_dbm_value));
	}
	if (button == ui->pushButton_power_watt_2){
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(power_watt_value),'f',6)).toDouble();
		serial_parser(serial_setPower_watt(SG_port, channel_select, power_watt_value));
	}

	// PSU parameters
	if (button == ui->pushButton_PSU_voltage_1){
		serial_parser(serial_setPSU_voltage_setpoint(SG_port, channel_select, PSU_voltage_setpoint));
	}
	if (button == ui->pushButton_PSU_current_limit_1){
		serial_parser(serial_setPSU_current_limit(SG_port, channel_select, PSU_current_limit));
	}

	//Sweep Page parameters
	if (button == ui->pushButton_SWP_power_dbm_1){
		SWP_power_watt = zeroChopper(QString::number(convert_dbm_to_watt(SWP_power_dbm),'f',6)).toDouble();
	}
	if (button == ui->pushButton_SWP_power_watt_1){
		SWP_power_dbm = zeroChopper(QString::number(convert_watt_to_dbm(SWP_power_watt),'f',6)).toDouble();
	}


	/* update all the button text */
	update_labels();

//	/* Make sure SGx power says '-', as it's unreliable after GCS/MCS has been used */
//	if (button == ui->pushButton_VGA_attenuation_1 || button == ui->pushButton_IQMod_magnitude_1)
//	{
//		ui->pushButton_power_SGx_dbm_1->setText("-");
//	}
}

/**********************************************************************************************************************************************************************************
 * OK MINMAX CHECK
 * *******************************************************************************************************************************************************************************/

/* Slightly differs from startup_check_minmax because with startup config there is no perspective of changes occuring.
 * ok_check_minmax makes use of *target_parameter to correct certain inputs better than startup_check_minmax
 * Additionally the PWM_duty_cycle is limited to 99, rather than 100. only the PWM_enable_button should be able to set DC value to 100% */
void MainWindow::ok_check_minmax(QPushButton *button)
{
	//CW PWR WATT
	if (power_watt_value > config->get_power_watt_max()){
		power_watt_value = zeroChopper(QString::number(config->get_power_watt_max(),'f',6)).toDouble();
	}
	else if (power_watt_value < config->get_power_watt_min()){
		power_watt_value = zeroChopper(QString::number(config->get_power_watt_min(),'f',6)).toDouble();
	}

	//CW PWR DBM
	if (power_dbm_value > convert_watt_to_dbm(config->get_power_watt_max())){
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_max()),'f',6)).toDouble();
	}
	else if(power_dbm_value < convert_watt_to_dbm(config->get_power_watt_min())){
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_min()),'f',6)).toDouble();
	}

	// Standalone Power dBm
	if (power_dbm_standalone_value > power_dbm_standalone_max)
	{
		power_dbm_standalone_value = power_dbm_standalone_max;
	}
	else if (power_dbm_standalone_value < power_dbm_standalone_min)
	{
		power_dbm_standalone_value = power_dbm_standalone_min;
	}

	//CW FREQUENCY MHZ
	if (freq_value > config->get_frequency_max()){
		freq_value = config->get_frequency_max();
	}
	else if (freq_value < config->get_frequency_min()){
		freq_value = config->get_frequency_min();
	}

	//CW PHASE
	if (phase_value > 360.0){
		phase_value = fmod(phase_value, 360.0);	//modulo with decimals...
	}
	else if (phase_value < 0){
		phase_value = 0;
	}

	//PWM FREQUENCY HZ
	if (PWM_freq_value < config->get_PWM_frequency_min()){
		PWM_freq_value = config->get_PWM_frequency_min();
	}
	if (PWM_freq_value > config->get_PWM_frequency_max()){
		PWM_freq_value = config->get_PWM_frequency_max();
	}

	//PWM DUTY CYCLE
	if (power_control_mode == POWER_CONTROL_NORMAL) //Only use limit if We're in normal control mode (Auto-gain enabled)
	{
		PWM_duty_cycle_min = PWM_freq_value * config->get_PWM_minimum_pulse_length_ns() / 10000000;
		double leftover = fmod(PWM_duty_cycle_min, 1.0);

		if (leftover > 0 && leftover < 0.5){
			PWM_duty_cycle_min = round(PWM_duty_cycle_min + 1);
		}
		else
		{
			PWM_duty_cycle_min = round(PWM_duty_cycle_min);
		}
	}
	else
	{
		PWM_duty_cycle_min = 1;
	}

	if (PWM_duty_cycle_value > 99){
		PWM_duty_cycle_value = 99;
	}
	else if (PWM_duty_cycle_value < PWM_duty_cycle_min)
	{
		show_notification("Duty cycle must exceed " + QString::number(PWM_duty_cycle_min) + "% at " + QString::number(PWM_freq_value) + "Hz PWM frequency.");
		PWM_duty_cycle_value = PWM_duty_cycle_min;
	}

	//DLL UPPER FREQUENCY
	if (button == ui->pushButton_DLL_frequency_limit_upper_1)
	{
		if (DLL_freqLimit_UP_value > config->get_frequency_max() * 0.000001){
			DLL_freqLimit_UP_value = config->get_frequency_max() * 0.000001;
		}
		else if (DLL_freqLimit_UP_value <= DLL_freqLimit_DOWN_value){
			DLL_freqLimit_UP_value = DLL_freqLimit_DOWN_value + 1;		//Avoid issues if lower freq = 2400 MHz
		}
	}

	//DLL LOWER FREQUENCY
	if (button == ui->pushButton_DLL_frequency_limit_lower_1)
	{
		if (DLL_freqLimit_DOWN_value < config->get_frequency_min() * 0.000001){
			DLL_freqLimit_DOWN_value = config->get_frequency_min() * 0.000001;
		}
		else if (DLL_freqLimit_DOWN_value >= DLL_freqLimit_UP_value){
			DLL_freqLimit_DOWN_value = DLL_freqLimit_UP_value - 1;	//Avoid issues if upper freq = 2500 MHz
		}
	}

	//DLL START FREQUENCY
	if (DLL_freq_start_value > DLL_freqLimit_UP_value){
		DLL_freq_start_value = DLL_freqLimit_UP_value;
	}
	else if (DLL_freq_start_value <= DLL_freqLimit_DOWN_value){
		DLL_freq_start_value = DLL_freqLimit_DOWN_value;
	}

	//DLL STEP FREQUENCY
	if (DLL_freq_step_value > (DLL_freqLimit_UP_value - DLL_freqLimit_DOWN_value)){
		DLL_freq_step_value = (DLL_freqLimit_UP_value - DLL_freqLimit_DOWN_value);
	}
	else if (DLL_freq_step_value < config->get_DLL_freqLimit_step()){
		DLL_freq_step_value = config->get_DLL_freqLimit_step();
	}

	//DLL THRESHOLD
	if (DLL_threshold_value < 0){
		DLL_threshold_value = 0;
	}
	else if (DLL_threshold_value > DLL_threshold_max){
		DLL_threshold_value = DLL_threshold_max;
	}

	//DLL MAIN DELAY
	if (DLL_main_delay_value < DLL_main_delay_min){
		DLL_main_delay_value = DLL_main_delay_min;
	}
	else if (DLL_main_delay_value > DLL_main_delay_max){
		DLL_main_delay_value = DLL_main_delay_max;
	}

	//The values only for a 2.45GHz SG Board.
	//ALL UPPER FREQUENCY
	if (button == ui->pushButton_ALL_frequency_limit_upper_1)
	{
		if (ALL_freqLimit_UP_value > config->get_ALL_frequency_max()){
			ALL_freqLimit_UP_value = config->get_ALL_frequency_max();
		}
		else if (ALL_freqLimit_UP_value <= ALL_freqLimit_DOWN_value){
			ALL_freqLimit_UP_value = ALL_freqLimit_DOWN_value + 0.01;		//Avoid issues if lower freq = 2400 MHz
		}
	}

	//ALL LOWER FREQUENCY
	if (button == ui->pushButton_ALL_frequency_limit_lower_1)
	{
		if (ALL_freqLimit_DOWN_value < config->get_ALL_frequency_min()){
			ALL_freqLimit_DOWN_value = config->get_ALL_frequency_min();
		}
		else if (ALL_freqLimit_DOWN_value >= ALL_freqLimit_UP_value){
			ALL_freqLimit_DOWN_value = ALL_freqLimit_UP_value - 0.01;	//Avoid issues if upper freq = 2500 MHz
		}
	}

	//ALL THRESHOLD
	if (ALL_threshold_value < 0){
		ALL_threshold_value = 0;
	}
	else if (ALL_threshold_value > 3.3)	{
		ALL_threshold_value = 3.3;
	}

	//VGA ATTENUATION FORWARD
	if (DVGA_attenuation_forward < 0){
		DVGA_attenuation_forward = 0;
	}
	else if (DVGA_attenuation_forward > 31.5){
		DVGA_attenuation_forward = 31.5;
	}

	//VGA ATTENUATION REFLECTED
	if (DVGA_attenuation_reflected < 0){
		DVGA_attenuation_reflected = 0;
	}
	else if (DVGA_attenuation_reflected > 31.5){
		DVGA_attenuation_reflected = 31.5;
	}

	//SWEEP START FREQUENCY
	if (button == ui->pushButton_SWP_start_frequency_1)
	{
		if (SWP_start_freq < config->get_frequency_min()){
			SWP_start_freq = config->get_frequency_min();
		}
		else if (SWP_start_freq >= SWP_stop_freq){
			SWP_start_freq = SWP_stop_freq - 1000000;	//Avoid issues if upper freq = 2500 MHz
		}
	}

	//SWEEP STOP FREQUENCY
	if (button == ui->pushButton_SWP_stop_frequency_1)
	{
		if (SWP_stop_freq > config->get_frequency_max()){
			SWP_stop_freq = config->get_frequency_max();
		}
		else if (SWP_stop_freq <= SWP_start_freq){
			SWP_stop_freq = SWP_start_freq + 1000000;	//Avoid issues if lower freq = 2400 MHz
		}
	}

	//SWEEP STEP FREQUENCY
	if (SWP_step_freq > 100000000.0){
		SWP_step_freq = 100000000.0;
	}
	else if ((((SWP_stop_freq - SWP_start_freq) / SWP_step_freq) + 1) > SWP_maximum_measure_points)
	{
		//These calculations are in Hz, instead of MHz, so everything is multiplied by 1.000.000
		SWP_step_freq = QString::number(((SWP_stop_freq - SWP_start_freq) / (SWP_maximum_measure_points - 1))/1000000, 'f', 2).toDouble() * 1000000;
		show_notification("Cannot sweep more than " + QString::number(SWP_maximum_measure_points) + " points in the frequency band.");
	}

	//SWEEP POWER WATT
	if (SWP_power_watt > config->get_power_watt_max()){
		SWP_power_watt = zeroChopper(QString::number(config->get_power_watt_max(),'f',6)).toDouble();
	}
	else if (SWP_power_watt < config->get_power_watt_min()){
		SWP_power_watt = zeroChopper(QString::number(config->get_power_watt_min(),'f',6)).toDouble();
	}

	//SWEEP POWER DBM
	if (SWP_power_dbm > convert_watt_to_dbm(config->get_power_watt_max())){
		SWP_power_dbm = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_max()),'f',6)).toDouble();
	}
	else if(SWP_power_dbm < convert_watt_to_dbm(config->get_power_watt_min())){
		SWP_power_dbm = zeroChopper(QString::number(convert_watt_to_dbm(config->get_power_watt_min()),'f',6)).toDouble();
	}

	if (power_control_mode == POWER_CONTROL_AMPLIFIER)
	{
		if (VGA_attenuation_value > config->get_AM_attenuation_max_db())
		{
			VGA_attenuation_value = config->get_AM_attenuation_max_db();
		}
		else if (VGA_attenuation_value < config->get_AM_attenuation_min_db())
		{
			VGA_attenuation_value = config->get_AM_attenuation_min_db();
		}
	}
}


/**********************************************************************************************************************************************************************************
 * MENU BUTTONS
 * *******************************************************************************************************************************************************************************/
void MainWindow::on_menu_power_Button_clicked()
{
	menuButton_manager(ui->menu_power_Button);
}
void MainWindow::on_menu_home_Button_clicked()
{
	menuButton_manager(ui->menu_home_Button);
}
void MainWindow::on_menu_ALL_Button_clicked()
{
	menuButton_manager(ui->menu_ALL_Button);
}
void MainWindow::on_menu_settings_Button_clicked()
{
	menuButton_manager(ui->menu_settings_Button);
}
void MainWindow::on_menu_sweep_Button_clicked()
{
	menuButton_manager(ui->menu_sweep_Button);
}
void MainWindow::on_menu_help_Button_clicked()
{
	menuButton_manager(ui->menu_help_Button);
}
void MainWindow::on_menu_DLL_Button_clicked()
{
	menuButton_manager(ui->menu_DLL_Button);
}
void MainWindow::on_menu_PSU_Button_clicked()
{
	menuButton_manager(ui->menu_PSU_Button);
}
//void MainWindow::on_menu_PID_Button_clicked()		//future?
//{
//	menuButton_manager(ui->menu_PID_Button);
//}


void MainWindow::menuButton_manager(QPushButton *menuButton_pointer)
{
	/* POWER / SHUTDOWN page */
	if (menuButton_pointer == ui->menu_power_Button)
	{
		if (RF_enabled == true){
			ui->pushButton_RF_enable_1->click();	//disable RF power because overview is lost
		}
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_page));
		last_page_index = ui->stackedWidget_3->currentIndex();
	}

	/* HOME page */
	if (menuButton_pointer == ui->menu_home_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->home_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* ALL page */
	if (menuButton_pointer == ui->menu_ALL_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->ALL_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* SETTINGS page */
	if (menuButton_pointer == ui->menu_settings_Button)
	{
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Settings_page));
		if (RF_enabled == true)
		{
			ui->pushButton_RF_enable_1->click();	//disable RF power because user loses sight of the active data.
		}
	}

	/* SWEEP page */
	if (menuButton_pointer == ui->menu_sweep_Button)
	{
		/* Only show execute SWP button if the Sweep page is selected */
		ui->pushButton_SWP_execute_1->setVisible(true);
		ui->pinkline_2->setVisible(true);

		if(ui->pushButton_SWP_execute_1->isChecked() == true)
		{
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Sweep_graph_page));
			if (RF_enabled == true)
			{
				ui->pushButton_RF_enable_1->click();	//disable RF power because user loses sight of the active data.
			}
		}
		else
		{
			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->sweep_controls_page));
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		}
	}
	else
	{
		/* Only show execute SWP button if the Sweep page is selected */
		ui->pushButton_SWP_execute_1->setVisible(false);
		ui->pinkline_2->setVisible(false);
	}

	/* HELP page */
	if (menuButton_pointer == ui->menu_help_Button)
	{
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Help_page));
	}

	/* DLL page */
	if (menuButton_pointer == ui->menu_DLL_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->DLL_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* PSU page */
	if (menuButton_pointer == ui->menu_PSU_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->PSU_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));

		/* Only poll PSU data if the PSU menu is selected */
		if (config->get_support_PSU_controls_enable())
		{
			PSUEG_parser();
		}
		if (config->get_support_PSU_controls_voltage())
		{
			PSUVG_parser();
		}
		if (config->get_support_PSU_controls_current_limit())
		{
			PSUIG_parser();
		}

		qDebug() << "PSU readings ON";
		connect(readings_timer, &QTimer::timeout, this, &MainWindow::get_PSU_IU);
		connect(readings_timer, &QTimer::timeout, this, &MainWindow::PSUEG_parser);
	}
	else
	{
		/* Only poll PSU data if the PSU menu is selected */
//		qDebug() << "PSU readings OFF";
		disconnect(readings_timer, &QTimer::timeout, this, &MainWindow::get_PSU_IU);
		disconnect(readings_timer, &QTimer::timeout, this, &MainWindow::PSUEG_parser);
	}

	/* PID page */
	//	if (menuButton_pointer == ui->menu_PID_Button)
	//	{

	//			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->PID_controls_page));
	//			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
	//			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	//	}
}


/**********************************************************************************************************************************************************************************
 * UTILITY BUTTONS
 * *******************************************************************************************************************************************************************************/
void MainWindow::on_pushButton_RF_enable_1_clicked()
{
	RF_enabled = !RF_enabled;
	serial_parser(serial_setRF_enable(SG_port, channel_select, RF_enabled));

	if (RF_enabled == true)
	{
		ui->pushButton_RF_enable_1->setText("RF is ON");
		if (config->get_continuous_readings_enabled() == false)
		{
			readings_timer->start(config->get_polling_rate_data());  //power readings timer
		}
	}
	else
	{
		ui->pushButton_RF_enable_1->setText("RF is OFF");
		if (config->get_continuous_readings_enabled() == false)
		{
			readings_timer->stop();
		}
	}

	/* Update LED status */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		LED_handler();
	}
#endif
}

void MainWindow::on_pushButton_unit_power_reflected_1_clicked()
{
	change_power_reading_units(ui->pushButton_unit_power_reflected_1);
}

void MainWindow::on_pushButton_unit_power_forward_1_clicked()
{
	change_power_reading_units(ui->pushButton_unit_power_forward_1);
}

void MainWindow::on_pushButton_unit_S11_1_clicked()
{
	change_power_reading_units(ui->pushButton_unit_S11_1);
}

void MainWindow::change_power_reading_units(QPushButton* button)
{
	if(button->isChecked() == true)
	{
		ui->pushButton_unit_power_reflected_1->setText("watt");
		ui->pushButton_unit_power_forward_1->setText("watt");
		ui->pushButton_unit_S11_1->setText("%");

		//instantly change the value to the right notation
		ui->label_power_reflected->setText(QString::number(convert_dbm_to_watt(PowerRFL_Value), 'f',1));
		ui->label_power_forward->setText(QString::number(convert_dbm_to_watt(PowerFRW_Value), 'f',1));

		if(PowerFRW_Value >= 30)
		{
			ui->label_s11_value->setText(QString::number((convert_dbm_to_watt(PowerRFL_Value)/convert_dbm_to_watt(PowerFRW_Value))*100, 'f',0));
		}
		else
		{
			ui->label_s11_value->setText("-");
		}

		ui->S11_title_label->setText("Reflection:");
		ui->pushButton_unit_S11_1->setChecked(true);
		ui->pushButton_unit_power_forward_1->setChecked(true);
		ui->pushButton_unit_power_reflected_1->setChecked(true);
	}
	else
	{
		ui->pushButton_unit_power_reflected_1->setText("dBm");
		ui->pushButton_unit_power_forward_1->setText("dBm");
		ui->pushButton_unit_S11_1->setText("dB");

		ui->label_power_reflected->setText(QString::number(PowerRFL_Value, 'f',1));
		ui->label_power_forward->setText(QString::number(PowerFRW_Value, 'f',1));

		if(PowerFRW_Value >= 30)
		{
			ui->label_s11_value->setText(QString::number((PowerRFL_Value - PowerFRW_Value), 'f',1));
		}
		else
		{
			ui->label_s11_value->setText("-");
		}
		ui->S11_title_label->setText("S11:");
		ui->pushButton_unit_S11_1->setChecked(false);
		ui->pushButton_unit_power_forward_1->setChecked(false);
		ui->pushButton_unit_power_reflected_1->setChecked(false);
	}
}

/**********************************************************************************************************************************************************************************
 * MODE BUTTONS
 * *******************************************************************************************************************************************************************************/

void MainWindow::on_pushButton_CW_enable_1_clicked()
{
	//
	//TODO:
	//Some improvement may be possible
	//

	if (power_control_mode != POWER_CONTROL_ANALOG_INPUT)
	{
		PWM_mode_enable(false);
	}
	ALL_mode_enable(false);
	DLL_mode_enable(false);
	if (power_control_mode != POWER_CONTROL_ANALOG_INPUT)
	{
		ui->pushButton_PWM_enable_1->setChecked(false);
	}
	ui->pushButton_ALL_enable_1->setChecked(false);
	ui->pushButton_DLL_enable_1->setChecked(false);

	// Don't allow user to pass inputs to items that aren't on the screen any longer.
	if (target_parameter_button == ui->pushButton_DLL_threshold_1 || target_parameter_button == ui->pushButton_DLL_step_frequency_1 ||
		target_parameter_button == ui->pushButton_ALL_threshold_1 ||
		target_parameter_button == ui->pushButton_PWM_duty_cycle_1 || target_parameter_button == ui->pushButton_PWM_frequency_1)
	{
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	}
	DVGA_enable();
}

void MainWindow::on_pushButton_PWM_enable_1_clicked()
{
	PWM_mode_enable(ui->pushButton_PWM_enable_1->isChecked());

	// Don't allow user to pass inputs to items that aren't on the screen any longer.
	if (target_parameter_button == ui->pushButton_PWM_duty_cycle_1 || target_parameter_button == ui->pushButton_PWM_frequency_1)
	{
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	}
}

void MainWindow::on_pushButton_PWM_enable_1_clicked(bool state)
{
	PWM_mode_enable(state);

	// Don't allow user to pass inputs to items that aren't on the screen any longer.
	if (target_parameter_button == ui->pushButton_PWM_duty_cycle_1 || target_parameter_button == ui->pushButton_PWM_frequency_1)
	{
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	}
}

void MainWindow::on_pushButton_ALL_enable_1_clicked()
{
	if(config->get_menu_ALL_enabled() == true){
		DLL_mode_enable(false);
		ALL_mode_enable(ui->pushButton_ALL_enable_1->isChecked());
		ui->pushButton_DLL_enable_1->setChecked(false);
	}

	// Don't allow user to pass inputs to items that aren't on the screen any longer.
	if (target_parameter_button == ui->pushButton_ALL_threshold_1 ||
		target_parameter_button == ui->pushButton_DLL_threshold_1 || target_parameter_button == ui->pushButton_DLL_step_frequency_1)
	{
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	}
	DVGA_enable();
}

void MainWindow::on_pushButton_DLL_enable_1_clicked()
{
	if(config->get_menu_ALL_enabled() == true){
		ALL_mode_enable(false);
	}
	DLL_mode_enable(ui->pushButton_DLL_enable_1->isChecked());
	ui->pushButton_ALL_enable_1->setChecked(false);

	// Don't allow user to pass inputs to items that aren't on the screen any longer.
	if (target_parameter_button == ui->pushButton_DLL_threshold_1 || target_parameter_button == ui->pushButton_DLL_step_frequency_1 ||
		target_parameter_button == ui->pushButton_ALL_threshold_1)
	{
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	}
	DVGA_enable();
}

void MainWindow::PWM_mode_enable(bool state)
{
	show_PWM_settings(state);
	serial_parser(serial_setPWM_enable(SG_port, channel_select, state, PWM_duty_cycle_value));
	PWM_enabled = state;

	/* configure analog input mode */
	if (ALL_enabled == true){
		AIS_modulation_source = 1;
	}
	else{
		AIS_modulation_source = 0;
	}
	Analog_input_mode_enable(power_control_mode, AIS_modulation_source);

	bool CW_status = !(ALL_enabled || DLL_enabled || PWM_enabled);
	if (ui->pushButton_CW_enable_1->isChecked() != CW_status)
	{
		ui->pushButton_CW_enable_1->setChecked(CW_status);
	}
}

void MainWindow::ALL_mode_enable(bool state)
{
	if(config->get_menu_ALL_enabled() == true)
	{
		show_ALL_settings(state);
		serial_parser(serial_setALL_enable(SG_port, channel_select, state));
		ALL_enabled = state;
	}
	bool CW_status = !(ALL_enabled || DLL_enabled || PWM_enabled);
	if (ui->pushButton_CW_enable_1->isChecked() != CW_status)
	{
		ui->pushButton_CW_enable_1->setChecked(CW_status);
	}
}

void MainWindow::DLL_mode_enable(bool state)
{
	show_DLL_settings(state);
	serial_parser(serial_setDLL_enable(SG_port, channel_select, state));
	ui->label_DLL_frequency_lock_1->setVisible(state);
	ui->label_DLL_frequency_lock_2->setVisible(state);
	ui->label_DLL_frequency_lock_3->setVisible(state);
	DLL_enabled = state;

	//I wonder if doing this connect action over and over doesn't cause the program to get slower every time?
	if(DLL_enabled == true)
	{
		connect(readings_timer, SIGNAL(timeout()), this, SLOT(DLL_frequency_indication()));
	}

	bool CW_status = !(ALL_enabled || DLL_enabled || PWM_enabled);
	if (ui->pushButton_CW_enable_1->isChecked() != CW_status)
	{
		ui->pushButton_CW_enable_1->setChecked(CW_status);
	}
}

#warning temporary delete later...					...never gonna happen is it?
void MainWindow::DVGA_enable()
{
	if (ALL_enabled == false)
	{
		if (config->get_support_DVGA() == true)
		{
			//BEFORE ALL is disabled, the DVGA's must be returned to safe default values! Otherwise the attached device could blow up!
			if (serial_parser(serial_set_DVGA_forward(SG_port, channel_select, config->get_default_DVGA_amplifier_forward_enabled(), config->get_default_DVGA_attenuation_forward())) == true)
			{
				ui->pushButton_DVGA_attenuation_forward_1->setText(zeroChopper(QString::number(config->get_default_DVGA_attenuation_forward(),'f',2)));
				if (config->get_default_DVGA_amplifier_forward_enabled() == true)
				{
					ui->pushButton_DVGA_amplifier_forward_1->setText("ON");
					ui->pushButton_DVGA_amplifier_forward_1->setChecked(true);
				}
				else
				{
					ui->pushButton_DVGA_amplifier_forward_1->setText("OFF");
					ui->pushButton_DVGA_amplifier_forward_1->setChecked(false);
				}

				ui->pushButton_DVGA_attenuation_forward_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
				ui->label_DVGA_attenuation_forward_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
				ui->pushButton_DVGA_amplifier_forward_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
				ui->label_DVGA_amplifier_forward_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
			}

			if (serial_parser(serial_set_DVGA_reflected(SG_port, channel_select, config->get_default_DVGA_amplifier_reflected_enabled(), config->get_default_DVGA_attenuation_reflected())) == true)
			{
				ui->pushButton_DVGA_attenuation_reflected_1->setText(zeroChopper(QString::number(config->get_default_DVGA_attenuation_reflected(),'f',2)));

				if (config->get_default_DVGA_amplifier_reflected_enabled() == true)
				{
					ui->pushButton_DVGA_amplifier_reflected_1->setText("ON");
					ui->pushButton_DVGA_amplifier_reflected_1->setChecked(true);
				}
				else
				{
					ui->pushButton_DVGA_amplifier_reflected_1->setText("OFF");
					ui->pushButton_DVGA_amplifier_reflected_1->setChecked(false);
				}

				ui->pushButton_DVGA_attenuation_reflected_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
				ui->label_DVGA_attenuation_reflected_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
				ui->pushButton_DVGA_amplifier_reflected_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
				ui->label_DVGA_amplifier_reflected_1->setEnabled(ui->pushButton_ALL_enable_1->isChecked());
			}
		}

		//Turn off the numpad if ALL is disabled and the user was messing with DVGA attentuation settings
		if (target_parameter_button == ui->pushButton_DVGA_attenuation_forward_1 || target_parameter_button == ui->pushButton_DVGA_attenuation_reflected_1)
		{
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
		}

		/* configure analog input mode to IQ MOD only */
		AIS_modulation_source = 0;
		Analog_input_mode_enable(power_control_mode, AIS_modulation_source);
	}
	else
	{
		if (config->get_support_DVGA() == true)
		{
			serial_parser(serial_set_DVGA_forward(SG_port, channel_select, DVGA_amplifier_forward_enabled, DVGA_attenuation_forward));
			serial_parser(serial_set_DVGA_reflected(SG_port, channel_select, DVGA_amplifier_reflected_enabled, DVGA_attenuation_reflected));
		}

		update_labels();

		if (DVGA_amplifier_forward_enabled == true)
		{
			ui->pushButton_DVGA_amplifier_forward_1->setText("ON");
			ui->pushButton_DVGA_amplifier_forward_1->setChecked(true);
		}
		else
		{
			ui->pushButton_DVGA_amplifier_forward_1->setText("OFF");
			ui->pushButton_DVGA_amplifier_forward_1->setChecked(false);
		}

		if (DVGA_amplifier_reflected_enabled == true)
		{
			ui->pushButton_DVGA_amplifier_reflected_1->setText("ON");
			ui->pushButton_DVGA_amplifier_reflected_1->setChecked(true);
		}
		else
		{
			ui->pushButton_DVGA_amplifier_reflected_1->setText("OFF");
			ui->pushButton_DVGA_amplifier_reflected_1->setChecked(false);
		}

		/* configure analog input mode to VGA only */
		AIS_modulation_source = 1;
		Analog_input_mode_enable(power_control_mode, AIS_modulation_source);
	}

	ui->pushButton_DVGA_attenuation_forward_1->setEnabled(	config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());
	ui->label_DVGA_attenuation_forward_1->setEnabled(		config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());
	ui->pushButton_DVGA_amplifier_forward_1->setEnabled(		config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());
	ui->label_DVGA_amplifier_forward_1->setEnabled(			config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());

	ui->pushButton_DVGA_attenuation_reflected_1->setEnabled(config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());
	ui->label_DVGA_attenuation_reflected_1->setEnabled(		config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());
	ui->pushButton_DVGA_amplifier_reflected_1->setEnabled(	config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());
	ui->label_DVGA_amplifier_reflected_1->setEnabled(		config->get_support_DVGA() && ui->pushButton_ALL_enable_1->isChecked());
}


//todo: maybe this functions should be relocated to the polling functions section
void MainWindow::DLL_frequency_indication()
{
	FCG_parser(config->get_console_output_enabled());
	ui->pushButton_frequency_1->setText(zeroChopper(QString::number(freq_value * 0.000001, 'f', 3)));
	ui->label_DLL_frequency_lock_2->setText(zeroChopper(QString::number(freq_value * 0.000001, 'f', 3)));

	if (DLL_enabled == false)
	{
		disconnect(readings_timer, SIGNAL(timeout()), this, SLOT(DLL_frequency_indication()));
	}

	/*
	QString get_string = serial_getFrequency(serial,channel_select, config->get_console_output_enabled());
	QRegExp regexp("\\$FCG,\\d+,(\\d+).(\\d+)\r\n");

	//Show frequency in GUI
	if (regexp.indexIn(get_string)>=0) {

		DLL_settle_frequency = (regexp.cap(1) + "." + regexp.cap(2)).toDouble() * 1000000;
		ui->pushButton_frequency_1->setText(zeroChopper(QString::number(DLL_settle_frequency * 0.000001, 'f', 3)));
		ui->label_DLL_frequency_lock_2->setText(zeroChopper(QString::number(DLL_settle_frequency * 0.000001, 'f', 3)));
		if(config->get_console_output_enabled() == true)
		{
			qDebug() << "freq_value" << freq_value << "\n";
		}
	}

	if (DLL_enabled == false)
	{
		//Set the frequency value back to normal again.
		freq_value = DLL_settle_frequency;
		serial_parser(serial_setFrequency(serial, channel_select, freq_value*0.000001));
		ui->pushButton_frequency_1->setText(zeroChopper(QString::number(freq_value * 0.000001, 'f', 3)));
		disconnect(timer, SIGNAL(timeout()), this, SLOT(DLL_frequency_indication()));
	}
	*/
}

/**********************************************************************************************************************************************************************************
 * Rotary Encoder Thread
 * *******************************************************************************************************************************************************************************/
#ifdef ROTARY_ENCODER_ENABLED

void MainWindow::thread_ErrorHandler1(QString){
	qDebug() << "Something went wrong in RE_thread";
	serialWriteRead(serial, "something went wrong in RE_thread");
}

void MainWindow::thread_Knob_enterHandler()
{
	if (menu_index != 2){
		menu_enter = !menu_enter;
		if (menu_enter == true){
			RE_menu_indicator();
		}
		else{
			RE_menu_indicator();	//run here to avoid visual lag of dealing with the serial
			RE_update_values();
		}
	}
	else {
		ui->RF_enableButton_2->click();
		RE_menu_indicator();
	}
}

void MainWindow::thread_Knob_leftHandler(){
	//do GUI stuff
	if(menu_enter == false){
		move_menu_index(-1);
	}
	else{
		switch(menu_index){
		case 0:
			freq_value -= 500000 * RE_freq_multiplier;
			if (freq_value < 2400000000){
				freq_value = 2400000000;
			}
			ui->minimalist_freq_label->setText(QString::number(freq_value * 0.000001, 'f', 2));
			break;
		case 1:
			power_watt_value -= 0.2 * RE_pwr_multiplier;
			if (power_watt_value < 0.2){
				power_watt_value = 0.2;
			}
			ui->minimalist_powerW_label->setText(QString::number(power_watt_value, 'f', 2));
			break;
		}
	}
}

void MainWindow::thread_Knob_rightHandler(){
	if(menu_enter == false){
		move_menu_index(+1);
	}
	else{
		switch(menu_index){
		case 0:
			freq_value += 500000 * RE_freq_multiplier;
			if (freq_value > 2500000000){
				freq_value = 2500000000;
			}
			ui->minimalist_freq_label->setText(QString::number(freq_value * 0.000001, 'f', 2));
			break;
		case 1:
			power_watt_value += 0.2 * RE_pwr_multiplier;

			if (power_watt_value > 500){
				power_watt_value = 500;
			}
			ui->minimalist_powerW_label->setText(QString::number(power_watt_value, 'f', 2));
			break;
		}
	}
}

void MainWindow::thread_ManualHandler(){
	//disable doing serial stuff
	uart.close();
	ui->minimalist_header->setText("Manual Mode");
	ui->RF_enableButton_2->setEnabled(true);
	if (RF_enabled == true){
		timer->start(power_get_refresh_rate);
	}
}

void MainWindow::thread_TransparentHandler(){
	menu_enter = false;
	ui->RF_enableButton_2->setEnabled(false);
	timer->stop();
	RE_update_timer->stop();

	if (uart.open(QIODevice::ReadWrite)) {
		qDebug() << "Serial Setup -> Connection Established";
		ui->minimalist_header->setText("Remote Command Mode");
	}
	else {
		QMessageBox message;
		message.critical(nullptr, "Serial connection error", "No connection found on serial0. Please try again.");
	}
}

void MainWindow::thread_AccelerationHandler(){
	RE_freq_multiplier = 5;
	RE_pwr_multiplier = 25;
}

void MainWindow::thread_DecelerationHandler(){
	RE_freq_multiplier = 1;
	RE_pwr_multiplier = 1;
}

void MainWindow::move_menu_index(int i){
	int localmin = 0;
	int localmax = 2;
	menu_index += i;
	if(menu_index > localmax){
		 menu_index = localmax;
	}
	else if(menu_index < localmin){
		 menu_index = localmin;
	}
	RE_menu_indicator();
}

void MainWindow::RE_menu_indicator(){
	 //reset
	 ui->minimalist_freq_label->setStyleSheet(
					 "background-color:rgb(187,224,254);");
	 ui->minimalist_label->setStyleSheet(
					 "background-color:rgb(187,224,254);");
	 ui->minimalist_label_7->setStyleSheet(
					 "background-color:rgb(187,224,254);");

	 ui->minimalist_powerW_label->setStyleSheet(
					 "background-color:rgb(162,213,254);");
	 ui->minimalist_label_2->setStyleSheet(
					 "background-color:rgb(162,213,254);");
	 ui->minimalist_label_8->setStyleSheet(
					 "background-color:rgb(162,213,254);");

	 ui->minimalist_label_3->setStyleSheet(
					 "background-color:rgb(162,213,254);");
	 ui->minimalist_label_12->setStyleSheet(
					 "background-color:rgb(162,213,254);");

	 if(menu_enter == false){
		  switch(menu_index){
		  case 0:
				ui->minimalist_freq_label->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				ui->minimalist_label->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				ui->minimalist_label_7->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				break;
		  case 1:
				ui->minimalist_powerW_label->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				ui->minimalist_label_2->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				ui->minimalist_label_8->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				break;
		  case 2:
				if(ui->RF_enableButton_2->isChecked() == false){
					 ui->minimalist_label_3->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
					 ui->minimalist_label_12->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				}
				else{
					 ui->minimalist_label_3->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
					 ui->minimalist_label_12->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				}
				break;
		  }
	 }
	 else{
		  switch(menu_index){
		  case 0:
				ui->minimalist_freq_label->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				ui->minimalist_label->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				ui->minimalist_label_7->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				break;
		  case 1:
				ui->minimalist_powerW_label->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				ui->minimalist_label_2->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				ui->minimalist_label_8->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				break;
		  case 2:
				if(ui->RF_enableButton_2->isChecked() == true){
					 ui->minimalist_label_3->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
					 ui->minimalist_label_12->setStyleSheet(
								"background-color:rgb(1,89,161);"
								"color:rgb(255,255,255);");
				}
				else{
					 ui->minimalist_label_3->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
					 ui->minimalist_label_12->setStyleSheet(
								"background-color:rgb(231,78,145);"
								"color:rgb(255,255,255);");
				}
				break;
		  }
	 }
}

void MainWindow::RE_update_values(){
	 switch(menu_index){
	 case 0:
		  serial_parser(serial_setFrequency(serial, channel_select, freq_value*0.000001));
		  qDebug() << freq_value*0.000001;
		  break;
	 case 1:
		  power_dbm_value = convert_watt_to_dbm(power_watt_value);
		  serial_set(Power_dbm(serial, channel_select, power_dbm_value));
		  //serial_parser(serial_setPower_watt(serial, channel_select, power_watt_value));
		  qDebug() << power_watt_value;
		  break;
	 }
}

void MainWindow::on_RF_enableButton_2_clicked()
{
	 RF_enabled = !RF_enabled;
	 serial_parser(serial_setRF_enable(serial, channel_select, RF_enabled));
	 if (RF_enabled == true){
		  ui->pushButton_RF_enable_1->setText("RF is ON");
		  ui->RF_enableButton_2->setText("RF is ON");
		  timer->start(power_get_refresh_rate);
	 }
	 else{
		  ui->pushButton_RF_enable_1->setText("RF is OFF");
		  ui->RF_enableButton_2->setText("RF is OFF");
		  timer->stop();
		  ui->label_power_forward->setText("-");
		  ui->label_power_reflected->setText("-");
		  ui->label_s11_value->setText("-");
		  ui->forward_power_label_2->setText("-");
		  ui->reflected_power_label_2->setText("-");
		  ui->S11_label_2->setText("-");
	 }
}
#endif


/**********************************************************************************************************************************************************************************
 * REMOTE COMMAND MODE
 * *******************************************************************************************************************************************************************************/
#ifdef ROTARY_ENCODER_ENABLED
void MainWindow::RCM_GUI_Updater(){
	switch(change_indicator){
	case 1:
		if(RF_enabled != ui->RF_enableButton_2->isChecked()){
			if(RF_enabled == true){
				ui->RF_enableButton_2->setChecked(true);
				ui->RF_enableButton_2->setText("RF is ON");
			}
			else if (RF_enabled == false){
				ui->RF_enableButton_2->setChecked(false);
				ui->RF_enableButton_2->setText("RF is OFF");
			}
		}
		break;
	case 2:
		if (freq_value > 2500000000 ){
			freq_value = 2500000000;
		}
		else if(freq_value < 2400000000){
			freq_value = 2400000000;
		}
		ui->minimalist_freq_label->setText(QString::number(freq_value * 0.000001, 'f', 2));
		break;
	case 3:
		if(power_dbm_value > 56.9897){
			power_dbm_value = 56.9897;
		}
		else if (power_dbm_value < 23.010299){
			power_dbm_value = 23.010299;
		}
		ui->minimalist_powerW_label->setText(QString::number(convert_dbm_to_watt(power_dbm_value), 'f', 2));
		break;
	case 4:
		if (power_watt_value > 500){
			power_watt_value = 500;
		}
		else if(power_watt_value < 0.2){
			power_watt_value = 0.2;
		}
		ui->minimalist_powerW_label->setText(QString::number(power_watt_value, 'f', 2));
		break;
	case 5:
		ui->reflected_power_label_2->setText(zeroChopper(QString::number(convert_dbm_to_watt(PowerRFL_Value), 'f',2)));
		ui->forward_power_label_2->setText(zeroChopper(QString::number(convert_dbm_to_watt(PowerFRW_Value), 'f',2)));
		ui->S11_label_2->setText(zeroChopper(QString::number((convert_dbm_to_watt(PowerRFL_Value)/convert_dbm_to_watt(PowerFRW_Value))*100, 'f',2)));
		break;
	case 6:
		ui->temperature_label_2->setText(zeroChopper(QString::number(Temperature_Value, 'f',2)));
		break;
	}
}
#endif

/* Remote Command Mode OFF */
void MainWindow::on_pushButton_remote_command_OFF_1_clicked()
{
	if (remote_command_mode != REMOTE_COMMAND_OFF)
	{
		RCM_controls_enable(true);
		TCP_disconnect_server();
		UART_disconnected();

		remote_command_mode = REMOTE_COMMAND_OFF;

		/* trigger a reset and clean up the startup junk text */
		if (config->get_reset_on_startup() == true)
		{
			port_open(&SG_port);
			safe_reset();
			reset_restore();
			ui->settings_plainTextEdit->appendPlainText(">\tSystem has been reset to defaults.");

			if (config->get_target_RF_enable() == true)
			{
				ui->settings_plainTextEdit->appendPlainText(">\tCaution: RF is auto-enabled");
			}
		}
		else
		{
			ui->settings_plainTextEdit->appendPlainText(">\tWarning: Reset is disabled. Settings remain unchanged.\n\tGUI does not accurately reflect the state of the SG Board!");
		}
	}

	/* Update LED status */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		LED_handler();
	}
#endif
}

/*  Remote Command Mode UART / USB */
void MainWindow::on_pushButton_remote_command_USB_1_clicked()
{
	if (remote_command_mode != REMOTE_COMMAND_USB)
	{
		/* Stop automated command spam in RCM */
		readings_timer->stop();

		/* Disconnected other UART-related RCM things */
		UART_disconnected();

		//Pick a port for RCM.
		QString RCM_portname;
		if (config->get_RCM_port_USB() != "")
		{
			RCM_portname = config->get_RCM_port_USB();
		}
		else
		{
			if (!get_RCM_port().isEmpty())
			{
				RCM_portname = get_RCM_port().at(0).portName();
			}
		}
		serial_Setup(RCM_port, RCM_portname);	//Port for external USB communication.

		if(RCM_port.open(QIODevice::ReadWrite) && SG_port.isOpen())
		{
			remote_command_mode = REMOTE_COMMAND_USB;
			RCM_controls_enable(false);
			TCP_disconnect_server();
			connect(&RCM_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Pass()));
			connect(&SG_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Return()));
			ui->settings_plainTextEdit->appendPlainText(QString(">\tUSB connection opened. Device is listening to %1.").arg(RCM_port.portName()));
		}
		else
		{
			ui->settings_plainTextEdit->appendPlainText(QString(">\tError - unable to open the USB connection:\n\t%1").arg(RCM_port.errorString()));
			ui->pushButton_remote_command_OFF_1->click(); //indicate that TM did not start.
			readings_timer->start(config->get_polling_rate_data());
		}
	}

	/* Update LED status */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		LED_handler();
	}
#endif
}

/*  Remote Command Mode UART / RS-232 */
void MainWindow::on_pushButton_remote_command_RS232_1_clicked()
{
	if (remote_command_mode != REMOTE_COMMAND_RS232)
	{
		/* Stop automated command spam in RCM */
		readings_timer->stop();

		/* Disconnected other UART-related RCM things */
		UART_disconnected();

		//Pick a port for RCM.
		QString RCM_portname = "serial0";

		serial_Setup(RCM_port, RCM_portname);	//Port for external USB communication.

		if(RCM_port.open(QIODevice::ReadWrite) && SG_port.isOpen())
		{
			remote_command_mode = REMOTE_COMMAND_RS232;
			RCM_controls_enable(false);
			TCP_disconnect_server();
			connect(&RCM_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Pass()));
			connect(&SG_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Return()));

			/* Set RS-232 enabled on HAT */
#if defined(Q_OS_LINUX)
			if (config->get_support_HAT_B14_0835())
			{
				GPIO_HAT_serial_mode->setValue("0");
			}
#endif

			ui->settings_plainTextEdit->appendPlainText(QString(">\tRS-232 connection opened. Device is listening to %1.").arg(RCM_port.portName()));
		}
		else
		{
			ui->settings_plainTextEdit->appendPlainText(QString(">\tError - unable to open the RS-232 connection:\n\t%1").arg(RCM_port.errorString()));
			ui->pushButton_remote_command_OFF_1->click(); //indicate that TM did not start.
			readings_timer->start(config->get_polling_rate_data());
		}
	}

	/* Update LED status */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		LED_handler();
	}
#endif
}

/*  Remote Command Mode UART / RS-485 */
void MainWindow::on_pushButton_remote_command_RS485_1_clicked()
{
	if (remote_command_mode != REMOTE_COMMAND_RS485)
	{
		/* Stop automated command spam in RCM */
		readings_timer->stop();

		/* Disconnected other UART-related RCM things */
		UART_disconnected();

		//Pick a port for RCM.
		QString RCM_portname = "serial0";

		serial_Setup(RCM_port, RCM_portname);	//Port for external USB communication.

		if(RCM_port.open(QIODevice::ReadWrite) && SG_port.isOpen())
		{
			remote_command_mode = REMOTE_COMMAND_RS485;
			RCM_controls_enable(false);
			TCP_disconnect_server();
			connect(&RCM_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Pass()));
			connect(&SG_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Return()));

			/* Set RS-485 enabled on HAT */
#if defined(Q_OS_LINUX)
			if (config->get_support_HAT_B14_0835())
			{
				GPIO_HAT_serial_mode->setValue("1");
			}
#endif
			ui->settings_plainTextEdit->appendPlainText(QString(">\tRS-485 connection opened. Device is listening to %1.").arg(RCM_port.portName()));
		}
		else
		{
			ui->settings_plainTextEdit->appendPlainText(QString(">\tError - unable to open the RS-485 connection:\n\t%1").arg(RCM_port.errorString()));
			ui->pushButton_remote_command_OFF_1->click(); //indicate that TM did not start.
			readings_timer->start(config->get_polling_rate_data());
		}
	}

	/* Update LED status */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		LED_handler();
	}
#endif
}

void MainWindow::UART_RCM_Pass()
{
	 //if NOT defined
	 #ifndef ROTARY_ENCODER_ENABLED
		  serial_RCM_Transmit(RCM_port,SG_port);
	 #endif

	 //Rotary encoder GUI can visualize certain $cmd's (ECS, FCS, PWRSGD, CPARS, PAG)
	 #ifdef ROTARY_ENCODER_ENABLED
		  serial_RCM_Receive(uart, serial, change_indicator, freq_value, RF_enabled, power_dbm_value, power_watt_value);
		  RCM_GUI_Updater();
	 #endif
}

void MainWindow::UART_RCM_Return()
{
#ifndef ROTARY_ENCODER_ENABLED		//if NOT defined
#if defined (Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835() && remote_command_mode == REMOTE_COMMAND_RS485)
	{
		serial_RCM_Receive_RS485(RCM_port, SG_port, *GPIO_RS485_RW_flipper);
	}
	else
	{
		serial_RCM_Receive(RCM_port, SG_port);
	}
#else

	serial_RCM_Receive(RCM_port, SG_port);

#endif	//Q_OS_LINUX
#endif	//ROTARY_ENCODER_ENABLED

	//Rotary encoder GUI can visualize certain $cmd's (ECS, FCS, PWRSGD, CPARS, PAG)
#ifdef ROTARY_ENCODER_ENABLED
	serial_RCM_Return(uart, serial, change_indicator, PowerFRW_Value, PowerRFL_Value, Temperature_Value);
	RCM_GUI_Updater();
#endif
}

/*  Remote Command Mode TCP */
void MainWindow::on_pushButton_remote_command_TCP_1_clicked()
{
	/* Without this check; when TCP button is pressed twice, it complains "the bound adress is already in use"
	 * It then fails to (re)initialize TCP RCM mode and cannot be started anymore afterwards! */
	if(remote_command_mode != REMOTE_COMMAND_TCP)
	{
		if(SG_port.isOpen())
		{
			/* Cannot have software spamming stuff while in RCM */
			readings_timer->stop();

			remote_command_mode = REMOTE_COMMAND_TCP;
			RCM_controls_enable(false);
			UART_disconnected();

			/* Open the TCP server */
			tcpServer = new QTcpServer(this);
			tcpSocket = new QTcpSocket(this);
			if(tcpServer->listen(QHostAddress::Any, 9001))
			{
				/* await a new connection to the IP. */
				connect(tcpServer, SIGNAL(newConnection()), this, SLOT(TCP_newConnection()));
				QList<QHostAddress> ipAddressesList = QNetworkInterface::allAddresses();
				for (int i = 0; i < ipAddressesList.size(); ++i)
				{
					// ToDo: create a network selection prompt. I don't think there's reasonably another way to choose the network.
					/* use the first non-localhost IPv4 address */
					if (ipAddressesList.at(i) != QHostAddress::LocalHost && ipAddressesList.at(i).toIPv4Address())
					{
						ipAddress = ipAddressesList.at(i).toString();
						break;
					}
				}
				if (ipAddress.isEmpty())	// if we did not find one, use IPv4 localhost
				{
					ipAddress = QHostAddress(QHostAddress::LocalHost).toString();
				}
				ui->settings_plainTextEdit->appendPlainText(QString(">\tTCP Server is listening for connections at:\n\tIP:\t\t%1\n\tPort:\t%2").arg(ipAddress).arg(tcpServer->serverPort()));
				qDebug() << "TCP remote command mode ENABLED\nIP:" << ipAddress << "\nport: " << tcpServer->serverPort();
			}
			else
			{
				ui->settings_plainTextEdit->appendPlainText(">\tError - unable to start tcpServer:\n\t" + tcpServer->errorString());
				ui->pushButton_remote_command_OFF_1->click(); //indicate that TM did not start.
				qDebug() << "Error: Unable to start tcpServer: " << tcpServer->errorString();
				readings_timer->start(config->get_polling_rate_data());
			}
		}
	}

	/* Update LED status */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		LED_handler();
	}
#endif
}

void MainWindow::TCP_newConnection()
{
	tcpSocket = tcpServer->nextPendingConnection();
	if(tcpSocket->state() == QTcpSocket::ConnectedState)
	{
		qDebug() << "New connection established.\n";
		qDebug() << tcpSocket->peerAddress();
		QString peer_address = tcpSocket->peerAddress().toString();
		QStringList peer_address_trim = peer_address.split(":");
		ui->settings_plainTextEdit->appendPlainText(QString(">\tNew connection established with [%1].").arg(peer_address_trim.last()));
	}
	connect(tcpSocket, SIGNAL(disconnected()), this, SLOT(TCP_client_disconnected()));
	connect(tcpSocket, SIGNAL(readyRead()), this, SLOT(TCP_RCM_Pass()));
	connect(&SG_port, SIGNAL(readyRead()), this, SLOT(TCP_RCM_Return()));
}

void MainWindow::TCP_RCM_Pass()
{
	serial_RCM_Transmit(*tcpSocket,SG_port);
}

void MainWindow::TCP_RCM_Return()
{
	serial_RCM_Receive(*tcpSocket,SG_port);
}


/* Disable the TCP server - Shut down TCP RCM mode */
void MainWindow::TCP_disconnect_server()
{
	disconnect(tcpSocket, SIGNAL(disconnected()));
	disconnect(tcpSocket, SIGNAL(readyRead()));
	disconnect(&SG_port, SIGNAL(readyRead()), this, SLOT(TCP_RCM_Return()));

	if (tcpServer->isListening() || tcpSocket->isOpen()){
		ui->settings_plainTextEdit->appendPlainText(">\tTCP Server has stopped listening for connections.");
		tcpServer->close();
		tcpSocket->close();
		qDebug() << "TCP Remote Command Mode DISABLED";
	}

	SG_port.close();
	SG_port.open(QIODevice::ReadWrite);
	//tcpSocket->deleteLater();

}

void MainWindow::TCP_client_disconnected()
{
	QString peer_address = tcpSocket->peerAddress().toString();
	QStringList peer_address_trim = peer_address.split(":");
	qDebug() << "Client disconnected from the server.";
	if(tcpServer->isListening())
	{
		ui->settings_plainTextEdit->appendPlainText(QString(">\tClient [%1] has disconnected.").arg(peer_address_trim.last()));
		ui->settings_plainTextEdit->appendPlainText(QString(">\tTCP Server is listening for connections at:\n\tIP:\t\t%1\n\tPort:\t%2").arg(ipAddress).arg(tcpServer->serverPort()));
	}
	else
	{
		ui->settings_plainTextEdit->appendPlainText(QString(">\tClient [%1] has been disconnected.").arg(peer_address_trim.last()));
	}
}

/* Close the UART connection - Shut down USB RCM mode */
void MainWindow::UART_disconnected()
{
	QString connection_type = "";
	switch(remote_command_mode)
	{
		case REMOTE_COMMAND_USB:
			connection_type = "USB";
			break;
		case REMOTE_COMMAND_RS232:
			connection_type = "RS-232";
			break;
		case REMOTE_COMMAND_RS485:
			connection_type = "RS-485";
			break;
		default:
			break;
	}

	disconnect(&RCM_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Pass()));
	disconnect(&SG_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Return()));
	if (RCM_port.isOpen())
	{
		RCM_port.close();
		if (connection_type != "")
		{
			ui->settings_plainTextEdit->appendPlainText(">\t"+ connection_type + " connection closed.");
		}
		else
		{
			ui->settings_plainTextEdit->appendPlainText(">\tConnection closed.");
		}
	}
	/* close and reopen serial port when sending a message to ensure a $RST didn't break comms */
	SG_port.close();
	SG_port.open(QIODevice::ReadWrite);

	QThread::msleep(1000);
	SG_port.clear();

	if (connection_type != "")
	{
		qDebug() << connection_type + " Remote Command Mode DISABLED";
	}
	else
	{
		qDebug() << "Remote Command Mode DISABLED";
	}
}

void MainWindow::RCM_controls_enable(bool state)
{
	ui->menu_home_Button->setEnabled(state);
	ui->menu_ALL_Button->setEnabled(state);
	ui->menu_DLL_Button->setEnabled(state);
	ui->menu_sweep_Button->setEnabled(state);
	ui->menu_PSU_Button->setEnabled(state);
	ui->menu_help_Button->setEnabled(state);
	ui->menu_settings_Button->setEnabled(state);
	ui->menu_power_Button->setEnabled(state);


	if(coherency_supported == true)
	{
		ui->pushButton_clock_source_standalone_1->setEnabled(state);
		ui->pushButton_clock_source_master_1->setEnabled(state);
		ui->pushButton_clock_source_slave_inline_1->setEnabled(state);
		ui->pushButton_clock_source_slave_end_1->setEnabled(state);
		ui->label_clock_source_1->setEnabled(state);
	}

	if(config->get_support_PWM_triggering() == true)
	{
		ui->pushButton_PWM_triggering_free_running_1->setEnabled(state);
		ui->pushButton_PWM_triggering_master_1->setEnabled(state);
		ui->pushButton_PWM_triggering_slave_1->setEnabled(state);
		ui->PWM_triggering_label->setEnabled(state);
	}

	if(config->get_support_power_control_modes() == true)
	{
		ui->pushButton_power_control_normal_1->setEnabled(state);
		ui->pushButton_power_control_analog_1->setEnabled(state && config->get_support_AIS_mode());
		ui->pushButton_power_control_feedforward_1->setEnabled(state && config->get_support_feedforward_control());
		ui->pushButton_power_control_amplifier_1->setEnabled(state && config->get_support_amplifier_mode());
		ui->label_power_control_mode_1->setEnabled(state);
	}

	if (config->get_support_external_triggering_mode() == true)
	{
		ui->pushButton_external_triggering_OFF_1->setEnabled(state);
		ui->pushButton_external_triggering_ON_1->setEnabled(state);
		ui->External_triggering_label->setEnabled(state);
	}
}

void MainWindow::RCM_mode_handler()
{
#warning defines are used here instead of RCM_MODE enum
	if (RCM_supported == true)
	{
		switch(config->get_remote_command_mode())
		{
			case REMOTE_COMMAND_OFF:
				ui->pushButton_remote_command_OFF_1->click();
				break;
			case REMOTE_COMMAND_USB:
				ui->menu_settings_Button->click();
				ui->pushButton_remote_command_USB_1->click();
				break;
			case REMOTE_COMMAND_TCP:
				ui->menu_settings_Button->click();
				ui->pushButton_remote_command_TCP_1->click();
				break;
			case REMOTE_COMMAND_RS232:
				ui->menu_settings_Button->click();
				ui->pushButton_remote_command_RS232_1->click();
				break;
			case REMOTE_COMMAND_RS485:
				ui->menu_settings_Button->click();
				ui->pushButton_remote_command_RS485_1->click();
				break;
			default:
				ui->pushButton_remote_command_OFF_1->click();
				break;
		}
	}
}

/**********************************************************************************************************************************************************************************
 * CHAIN MODE
 * *******************************************************************************************************************************************************************************/
void MainWindow::on_pushButton_clock_source_standalone_1_clicked()
{
	if(clock_source != 0)
	{
		clock_source = 0;
		serial_parser(serial_set_clock_source(SG_port, channel_select, clock_source));
	}
}

void MainWindow::on_pushButton_clock_source_master_1_clicked()
{
	if(clock_source != 1)
	{
		clock_source = 1;
		serial_parser(serial_set_clock_source(SG_port, channel_select, clock_source));
	}
}

void MainWindow::on_pushButton_clock_source_slave_inline_1_clicked()
{
	if(clock_source != 3)
	{
		clock_source = 3;
		serial_parser(serial_set_clock_source(SG_port, channel_select, clock_source));
	}
}

void MainWindow::on_pushButton_clock_source_slave_end_1_clicked()
{
	if(clock_source != 2)
	{
		clock_source = 2;
		serial_parser(serial_set_clock_source(SG_port, channel_select, clock_source));
	}
}

void MainWindow::Coherency_mode_handler(){
	//Set clock source / reference oscillator
	if (coherency_supported == true)
	{
		switch(config->get_coherency_mode())
		{
			case 0:
				ui->pushButton_clock_source_standalone_1->click();
				break;
			case 1:
				ui->pushButton_clock_source_master_1->click();
				break;
			case 2:
				ui->pushButton_clock_source_slave_end_1->click();
				break;
			case 3:
				ui->pushButton_clock_source_slave_inline_1->click();
				break;				
			default:
				ui->pushButton_clock_source_standalone_1->click();
				break;
		}
	}
	else
	{
		serial_parser(serial_set_clock_source(SG_port, channel_select, 0));
	}
}

/**********************************************************************************************************************************************************************************
 * PWM TRIGGERING
 * *******************************************************************************************************************************************************************************/
void MainWindow::on_pushButton_PWM_triggering_free_running_1_clicked()
{
	if (PWM_trigger_mode != 1)
	{
		PWM_trigger_mode = 1;
		serial_parser(serial_setPWM_triggering(SG_port, channel_select, PWM_trigger_mode, PWM_slave_port, PWM_slave_pin, PWM_master_port, PWM_master_pin));
	}
}

void MainWindow::on_pushButton_PWM_triggering_master_1_clicked()
{
	if (PWM_trigger_mode != 2)
	{
		PWM_trigger_mode = 2;
		serial_parser(serial_setPWM_triggering(SG_port, channel_select, PWM_trigger_mode, PWM_slave_port, PWM_slave_pin, PWM_master_port, PWM_master_pin));
	}
}

void MainWindow::on_pushButton_PWM_triggering_slave_1_clicked()
{
	if (PWM_trigger_mode != 3)
	{
		PWM_trigger_mode = 3;
		serial_parser(serial_setPWM_triggering(SG_port, channel_select, PWM_trigger_mode, PWM_slave_port, PWM_slave_pin, PWM_master_port, PWM_master_pin));
	}
}

void MainWindow::PWM_triggering_mode_handler()
{
	//Set PWM triggering
	if (config->get_support_PWM_triggering() == true)
	{
		switch(config->get_PWM_triggering_mode())
		{
			case 1:
				ui->pushButton_PWM_triggering_free_running_1->click();
				break;
			case 2:
				ui->pushButton_PWM_triggering_master_1->click();
				break;
			case 3:
				ui->pushButton_PWM_triggering_slave_1->click();
				break;
			default:
				ui->pushButton_PWM_triggering_free_running_1->click();
				break;
		}
	}
	else
	{
		serial_parser(serial_setPWM_triggering(SG_port, channel_select, 1, PWM_slave_port, PWM_slave_pin, PWM_master_port, PWM_master_pin));
	}
}

/**********************************************************************************************************************************************************************************
 * POWER CONTROL MODE
 * *******************************************************************************************************************************************************************************/
/* Set power control mode Normal: Autogain Enabled */
void MainWindow::on_pushButton_power_control_normal_1_clicked()
{
	power_control_mode = POWER_CONTROL_NORMAL;

	/* 4x250W ZHL with Raspi HAT -> switch to Generator mode */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external->setValue("0");
		GPIO_control_SGx->setValue("1");
	}
#endif

	/* Disable Analog Input Mode */
	Analog_input_mode_enable(power_control_mode, AIS_modulation_source);

	/* After exiting AIS, re-enable Autogain */
	serial_parser(serial_setAutoGain_enable(SG_port,channel_select,true));

	/* re-enable power controls */
	show_autogain_controls(true);
	show_power_dbm_standalone_controls(false);
	show_GCS_controls(false);
	show_MCS_controls(false);
	show_SGx_Power_controls(false);
	show_Amplifier_mode_controls(false);

	//After returning to normal power control mode ensure the PWM duty Cycle is updated to the proper value
	ok_check_minmax(ui->pushButton_PWM_duty_cycle_1);
	serial_parser(serial_setPWM_enable(SG_port, channel_select, PWM_enabled, PWM_duty_cycle_value));
	PWM_DCG_parser();
}

/* Set AIS mode enable; IQMod or VGA only; effectively a voltage controlled $GCS or $MCS depending on modulation source */
void MainWindow::on_pushButton_power_control_analog_1_clicked()
{
	/* Enable Analog Input Mode */
	power_control_mode = POWER_CONTROL_ANALOG_INPUT;

	/* 4x250W ZHL with Raspi HAT -> switch to Generator mode */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external->setValue("0");
		GPIO_control_SGx->setValue("1");
	}
#endif

	/* Enable Analog Input Mode */
	Analog_input_mode_enable(power_control_mode, AIS_modulation_source);

	/* disable power controls */
	show_autogain_controls(false);
	show_power_dbm_standalone_controls(false);
	show_GCS_controls(true);
	show_MCS_controls(true);
	show_SGx_Power_controls(false);
	show_Amplifier_mode_controls(false);

	ui->settings_plainTextEdit->appendPlainText(">\tVGA attenuation set to default value of " + QString::number(config->get_AIS_attenuation_value()) + "dB.");

	//
	// TODO:
	// The GCS/MCS availability should depend on the enable state of ALL...
	//
}

/* Set power control mode Feed Forward: MCS, GCS and SGx Power only */
void MainWindow::on_pushButton_power_control_feedforward_1_clicked()
{
	/* Enable Feedforward / Manual input Mode */
	power_control_mode = POWER_CONTROL_FEED_FWD;

	/* 4x250W ZHL with Raspi HAT -> switch to Generator mode */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external->setValue("0");
		GPIO_control_SGx->setValue("1");
	}
#endif

	/* Disable Analog Input Mode */
	Analog_input_mode_enable(false, AIS_modulation_source);
	serial_parser(serial_setAutoGain_enable(SG_port,channel_select, false));

	/* Manage power controls */
	if (PA_type == 1)	//PA type 1 uses a unique power control method; use $PWRDS to set standalone SG power.
	{
		/* Standalone SG FFWD mode - Uses a calibration written in EEPROM command sequence.
		 * Typically contains: $IQICS, $VGACCS, $FFCCS
		 *
		 * GUI behaviour:
		 * Hide (auto-gain) $PWRDS {X dBm, Y dBm);
		 * Hide (auto-gain) $PWRS
		 * Show (standalone) $PWRDS {-30 dBm, +30 dBm};
		 * Hide $PWRSGDS
		 * Show $GCS // but don't recommend use, as frequency optimization will overwrite anything you provide.
		 * Show $MCS // but don't recommend use, as frequency optimization will overwrite anything you provide.
		 */

		show_autogain_controls(false);
		show_power_dbm_standalone_controls(true);
		show_SGx_Power_controls(false);
		show_GCS_controls(true);
		show_MCS_controls(true);
		show_Amplifier_mode_controls(false);
	}
	else if (PA_type == 17 || PA_type == 24)
	{
		/* Calibrated SG FFWD mode - Uses calibrations written in the FeedForward calibration slots of the EEPROM.
		 *
		 * GUI behaviour:
		 * Show (auto-gain) $PWRDS {X dBm, Y dBm);
		 * Show (auto-gain) $PWRS
		 * Hide (standalone) $PWRDS {-30 dBm, +30 dBm};
		 * Hide $PWRSGDS
		 * Show $GCS // but don't recommend use, as frequency optimization will overwrite anything you provide.
		 * Show $MCS // but don't recommend use, as frequency optimization will overwrite anything you provide.
		 */

		show_autogain_controls(true);
		show_power_dbm_standalone_controls(false);
		show_SGx_Power_controls(false);
		show_GCS_controls(true);
		show_MCS_controls(true);
		show_Amplifier_mode_controls(false);
	}
	else
	{
		/* Basic FFWD mode - No calibrations, a vague approximation available with $PWRSGDS but not recommended;
		 *
		 * GUI behaviour:
		 * Hide (auto-gain) $PWRDS {X dBm, Y dBm);
		 * Hide (auto-gain) $PWRS
		 * Hide (standalone) $PWRDS {-30 dBm, +30 dBm};
		 * Show $PWRSGDS // But usage is not recommended.
		 * Show $GCS // Preferred method of control
		 * Show $MCS // Preferred method of control
		 */

		show_autogain_controls(false);
		show_power_dbm_standalone_controls(false);
		show_SGx_Power_controls(true);
		show_GCS_controls(true);
		show_MCS_controls(true);
		show_Amplifier_mode_controls(false);
	}

	PWRDG_parser();
	PWRSGDG_parser();
	MCG_parser();
	GCG_parser();

	ui->pushButton_power_dbm_standalone_1->setText("-");
	ui->pushButton_power_SGx_dbm_1->setText("-");
	ui->pushButton_VGA_attenuation_1->setText(zeroChopper(QString::number(VGA_attenuation_value,'f',1)));
	ui->pushButton_IQMod_magnitude_1->setText(zeroChopper(QString::number(magnitude_value,'f',1)));
}

/* Set power control mode Amplifier mode: GCS only */
void MainWindow::on_pushButton_power_control_amplifier_1_clicked()
{
	power_control_mode = POWER_CONTROL_AMPLIFIER;
	Analog_input_mode_enable(false, AIS_modulation_source);
	serial_parser(serial_setAutoGain_enable(SG_port,channel_select, false));

	/* Disable power controls */
	show_autogain_controls(false);
	show_power_dbm_standalone_controls(false);

	/* Enter CW mode; Turn off PWM / DLL / ALL on SGx Board */
	ui->pushButton_CW_enable_1->click();

	/* 4x250W ZHL with Raspi HAT; RF completely circumvents SGx board -> no point in having GCS
	 * Otherwise -> Show GCS controls */
	show_GCS_controls(true);
	show_MCS_controls(false);
	show_SGx_Power_controls(false);
	show_Amplifier_mode_controls(true);

//	PWRSGDG_parser();
	MCG_parser();
	GCG_parser();

	ui->pushButton_power_SGx_dbm_1->setText("-"); //zeroChopper(QString::number(sgx_power_value,'f',3)));
	ui->pushButton_VGA_attenuation_1->setText(zeroChopper(QString::number(VGA_attenuation_value,'f',1)));
	ui->pushButton_IQMod_magnitude_1->setText(zeroChopper(QString::number(magnitude_value,'f',1)));

	/* 4x250W ZHL with Raspi HAT -> switch to Amplifier mode */
#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external->setValue("1");
		GPIO_control_SGx->setValue("0");
	}
#endif
}

void MainWindow::Analog_input_mode_enable(int state, int mode)
{
	bool enable = false;

	//Make sure signal doesn't get duplicate connections...
	disconnect(readings_timer, &QTimer::timeout, this, &MainWindow::get_IQmodVGA);

	if (state == 1)
	{
		enable = true;
		connect(readings_timer, &QTimer::timeout, this, &MainWindow::get_IQmodVGA);
	}

	if (mode == 1)	//Modulation source = VGA
	{
		serial_parser(serial_setAnalogInput(SG_port,channel_select, enable, 1, config->get_AIS_attenuation_max_db(), config->get_AIS_ADC_min(), config->get_AIS_attenuation_min_db(), config->get_AIS_ADC_max()));
		//
		//TODO:
		//Technically if we're doing this at all, there should be an MCS being set here.
		//... or not at all cause there's a button on the screen for it now?
		//
	}
	else			//Modulation source = IQ mod
	{
		serial_parser(serial_setAnalogInput(SG_port,channel_select, enable, 0, config->get_AIS_magnitude_min(), config->get_AIS_ADC_min(), config->get_AIS_magnitude_max(), config->get_AIS_ADC_max()));
		if (enable == true)
		{
			//
			//TODO:
			//This should probably be switched to using the feed-forward attenuation value instead...
			//or not at all cause there's a button on the screen for it now?
			//
			serial_parser(serial_setVGA_attenuation(SG_port, channel_select, config->get_AIS_attenuation_value()));
		}
	}
}

void MainWindow::Power_control_mode_handler()
{
	if (config->get_support_power_control_modes() == true)
	{
		switch(power_control_mode)
		{
			case 0:
				ui->pushButton_power_control_normal_1->click();
				break;
			case 1:
				ui->pushButton_power_control_analog_1->click();
				break;
			case 2:
				ui->pushButton_power_control_feedforward_1->click();
				break;
			case 3:
				ui->pushButton_power_control_amplifier_1->click();
				break;
			default:
				ui->pushButton_power_control_normal_1->click();
				break;
		}
	}
	else
	{
		//If power control modes are not supported, set normal / autogain as the default
		on_pushButton_power_control_normal_1_clicked();
	}
}

/**********************************************************************************************************************************************************************************
 * External Triggering Mode
 * *******************************************************************************************************************************************************************************/
/* External triggering mode disabled; Enable PWM controls and set duty cycle to 100% */
void MainWindow::on_pushButton_external_triggering_OFF_1_clicked()
{
	//TODO: find a way to make this not do redundant actions, without breaking reset behaviour.
	ext_trig_mode = 0;
	External_triggering_enable(false);
	if (ext_trig_mode == 0 && power_control_mode != 3)
	{
		ui->menu_sweep_Button->setEnabled(true);
	}
}

/* External triggering mode enabled; Disable PWM controls and set duty cycle to 0% so that we can do external triggering */
void MainWindow::on_pushButton_external_triggering_ON_1_clicked()
{
	ext_trig_mode = 1;
	External_triggering_enable(true);
	ui->menu_sweep_Button->setEnabled(false);
}

void MainWindow::External_triggering_enable(bool state)
{
	/* Disable PWM controls so that we can do external triggering */
	if(state == true)
	{
		/* Disable PWM on screen (and hide PWM controls) and set PWM to 0% on SGx board */
		if(PWM_enabled == true)
		{
			on_pushButton_PWM_enable_1_clicked(false);
		}
		serial_parser(serial_setPWM_DC(SG_port,channel_select, 0));
	}
	else
	{
		/* set PWM to 100% on SGx board (CW) and do a DCG to configure the on-screen settings right */
		serial_parser(serial_setPWM_DC(SG_port,channel_select,100));
//		PWM_DCG_parser();
		PWM_DCG_duty_cycle_parser();
	}

	ui->pushButton_power_control_amplifier_1->setEnabled(!state);

	/* Workaround for amplifier mode to get the visuals right */
	if (power_control_mode != POWER_CONTROL_AMPLIFIER)
	{
		ui->pushButton_PWM_enable_1->setEnabled(!state);
	}
}

void MainWindow::External_triggering_handler()
{
	switch(ext_trig_mode)
	{
		case 0:
			ui->pushButton_external_triggering_OFF_1->setChecked(true);
			on_pushButton_external_triggering_OFF_1_clicked();
			break;
		case 1:
			ui->pushButton_external_triggering_ON_1->setChecked(true);
			on_pushButton_external_triggering_ON_1_clicked();
			break;
		default:
			ui->pushButton_external_triggering_OFF_1->setChecked(true);
			on_pushButton_external_triggering_OFF_1_clicked();
			break;
	}
}

/**********************************************************************************************************************************************************************************
 * SWEEP
 * *******************************************************************************************************************************************************************************/

void MainWindow::on_pushButton_SWP_execute_1_clicked()
{
	/* Why the !@#$ won't this show properly?! */
	ui->pushButton_SWP_execute_1->setText("Scanning");
	ui->pushButton_SWP_execute_2->setText("Scanning");

	if (ui->pushButton_unit_S11_1->isChecked() == true)
	{
		SWP_mode = "watt";
		ui->pushButton_SWP_unit->setText("Unit: linear");
	}
	else
	{
		SWP_mode = "dbm";
		ui->pushButton_SWP_unit->setText("Unit: log");
	}

	if (SWP_run_sweep() == 0)
	{
		SWP_draw_plot();
		ui->pushButton_SWP_execute_2->setText("Sweep");
		ui->pushButton_SWP_execute_1->setText("Sweep");
		ui->pushButton_SWP_execute_2->setChecked(false);

		if (RF_enabled == true){
			ui->pushButton_RF_enable_1->click();	//disable RF power after sweep is complete, because user loses sight of the active data.
		}
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Sweep_graph_page));
	}
	else
	{
		qDebug () << "bad data";
		show_notification("Invalid sweep data");
		ui->pushButton_SWP_execute_1->setChecked(false);
		ui->pushButton_SWP_execute_1->setText("Sweep");
		ui->pushButton_SWP_execute_2->setChecked(false);
		ui->pushButton_SWP_execute_2->setText("Sweep");
	}
}

void MainWindow::on_pushButton_SWP_unit_clicked()
{
	if(SWP_mode == "dbm")
	{
		SWP_mode = "watt";
		ui->pushButton_SWP_unit->setText("Unit: linear");
	}
	else
	{
		SWP_mode = "dbm";
		ui->pushButton_SWP_unit->setText("Unit: log");
	}
	SWP_draw_plot();
}


void MainWindow::on_pushButton_SWP_execute_2_clicked()
{
	ui->pushButton_SWP_execute_2->setText("Scanning");
	if (SWP_run_sweep() == 0)
	{
		SWP_draw_plot();
		ui->pushButton_SWP_execute_2->setText("Sweep");
		ui->pushButton_SWP_execute_2->setChecked(false);
	}
	else{
		qDebug () << "bad data";
	}
}

void MainWindow::on_pushButton_SWP_back_clicked()
{
	ui->pushButton_SWP_execute_1->setChecked(false);

	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->sweep_controls_page));
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->logo_page));
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
}

int MainWindow::SWP_run_sweep()
{
	QString SWP_raw_data = serial_runSweep_dbm(SG_port,channel_select,SWP_start_freq *0.000001,SWP_stop_freq *0.000001,SWP_step_freq *0.000001,SWP_power_dbm);
	QStringList SWP_data;
	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		SWP_data = SWP_raw_data.split("\r\n");
		//Remove the OK entry + the empty line that comes after.
		SWP_data.removeLast();
		SWP_data.removeLast();
//		qDebug() << "Stringlist: " << SWP_data;
	}
	else
	{
		if (SWP_raw_data != "")
		{
			QMessageBox message;
			message.warning(nullptr, "Communication Error", "Function returned Error:\n" + SWP_raw_data);
		}
		return -1;
	}

	if (SWP_data.count() <= 0)
	{
		return -1;
	}

	//Resize the vectors for the correct amount of data
	SWP_freq_data.resize(SWP_data.count());
	SWP_frw_data.resize(SWP_data.count());
	SWP_rfl_data.resize(SWP_data.count());
	SWP_s11_dbm_data.resize(SWP_data.count());
	SWP_s11_watt_data.resize(SWP_data.count());

	QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(-?\\d+.?\\d+),(-?\\d+.?\\d+)");
	for (int i = 0; i < SWP_data.count(); i++)
	{
		 qDebug() << SWP_data.at(i) << "#" << i;
		 if (regexp.indexIn(SWP_data.at(i))>=0)
		 {
			  QString string1 = regexp.cap(1);
			  QString string2 = regexp.cap(2);
			  QString string3 = regexp.cap(3);

			  SWP_freq_data[i] = string1.toDouble();
			  SWP_frw_data[i] = string2.toDouble();
			  SWP_rfl_data[i] = string3.toDouble();

			  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
			  SWP_s11_watt_data[i] = (convert_dbm_to_watt(SWP_rfl_data[i]) / convert_dbm_to_watt(SWP_frw_data[i])) * 100;
			  /* Perhaps theoretically a division by zero is possible here, but the minimum power value that can be set is 0.1W / 20dBm anyway...
			   * Since we get our power values in dBm and convert to watt, there's no such thing as 0 dbm anyway... therefor in practice divide by zero doesn't occur. */
		 }
	}
	return 0;
}

void MainWindow::SWP_draw_plot()
{
	ui->SWP_plot->addGraph();
	ui->SWP_plot->xAxis->setLabel("Frequency (MHz)");
	ui->SWP_plot->xAxis->setRange(SWP_freq_data[0],SWP_freq_data[SWP_freq_data.size()-1]);
	ui->SWP_plot->yAxis->setNumberFormat("f");
	ui->SWP_plot->yAxis->setNumberPrecision(2);
	double min_val, max_val;

	if(SWP_mode == "dbm")
	{
		ui->SWP_plot->graph(0)->setData(SWP_freq_data,SWP_s11_dbm_data);

		min_val = *std::min_element(SWP_s11_dbm_data.constBegin(),SWP_s11_dbm_data.constEnd());
		if (min_val > 0){
			 min_val = 0;
		}
		max_val = *std::max_element(SWP_s11_dbm_data.constBegin(),SWP_s11_dbm_data.constEnd());
		if(max_val < 0){
			 max_val = 0;
		}

		ui->SWP_plot->yAxis->setRange(min_val*1.1, max_val*1.1);
		ui->SWP_plot->yAxis->setLabel("S11 (dB)");
		//ui->SWP_plot->yAxis->setRangeReversed(true);
	}
	else
	{
		ui->SWP_plot->graph(0)->setData(SWP_freq_data,SWP_s11_watt_data);

		max_val = *std::max_element(SWP_s11_watt_data.constBegin(),SWP_s11_watt_data.constEnd());

		if(max_val <= 100)
		{
			max_val = 100;
		}
		ui->SWP_plot->yAxis->setRange(0,max_val);
		ui->SWP_plot->yAxis->setLabel("Reflection (%)");
	}

	ui->SWP_plot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom | QCP::iSelectItems);
	ui->SWP_plot->replot();
}


/**********************************************************************************************************************************************************************************
 * PSU
 * *******************************************************************************************************************************************************************************/

/* Configure / reset PSU to a default state according to config file */
void MainWindow::handler_PSU_reset_to_default()
{
	if (config->get_PSU_count() > 0)
	{
		/* If the RPi HAT is present, only write the PSU commands if GPIO PSU enable button is in a pressed state */
#if defined(Q_OS_LINUX)
		if (config->get_support_HAT_B14_0835())
		{
			if (GPIO_PSU_button_enable->readValue() > 0)		//If the front panel button is pressed.
			{
				#warning for some reason the switch readout does not go as aspected here  on the actual 1KW system. This if statement fails even though the button is pressed.

				/* Only configure the PSU(s) if valid inputs are defined in config, otherwise don't do anything / let the SGx EEPROM handle it instead. */
				if (config->get_PSU_voltage() > 0)
				{
					if (serial_parser(serial_setPSU_voltage_setpoint(SG_port, channel_select, config->get_PSU_voltage())) == true)
					{
						PSU_voltage_setpoint = config->get_PSU_voltage();
					}
				}
				if (config->get_PSU_current_limit() > 0)
				{
					if (serial_parser(serial_setPSU_current_limit(SG_port, channel_select, config->get_PSU_current_limit())) == true)
					{
						PSU_current_limit = config->get_PSU_current_limit();
					}
				}
				if (config->get_PSU_enabled() >= 0)
				{
					if (serial_parser(serial_setPSU_enable(SG_port, channel_select, config->get_PSU_enabled())) == true)
					{
						for(int i = 0; i < PSU_count; i++)
						{
							PSU_enable[i] = config->get_PSU_enabled();
						}
						PSU_enable_combined = config->get_PSU_enabled();
					}
				}
			}

			// Done. Exit the function early.
			return;
		}
#endif

		/* If there is no RPi HAT, then just send the commands so long as PSUs are configured */
		/* Only configure the PSU(s) if valid inputs are defined in config, otherwise don't do anything / let the SGx EEPROM handle it instead. */
		if (config->get_PSU_voltage() > 0)
		{
			if (serial_parser(serial_setPSU_voltage_setpoint(SG_port, channel_select, config->get_PSU_voltage())) == true)
			{
				PSU_voltage_setpoint = config->get_PSU_voltage();
			}
		}
		if (config->get_PSU_current_limit() > 0)
		{
			if (serial_parser(serial_setPSU_current_limit(SG_port, channel_select, config->get_PSU_current_limit())) == true)
			{
				PSU_current_limit = config->get_PSU_current_limit();
			}
		}
		if (config->get_PSU_enabled() >= 0)
		{
			if (serial_parser(serial_setPSU_enable(SG_port, channel_select, config->get_PSU_enabled())) == true)
			{
				for(int i = 0; i < PSU_count; i++)
				{
					PSU_enable[i] = config->get_PSU_enabled();
				}
				PSU_enable_combined = config->get_PSU_enabled();
			}
		}
	}
}

void MainWindow::get_PSU_IU()
{
	PSU_total_power = 0;
	QString get_string = serial_getPSU_IU_All_Reading(SG_port, channel_select, config->get_console_output_enabled());
	get_string = get_string.remove("\r");
	get_string = get_string.remove("\n");

	QStringList get_list = get_string.split(",");
	if (get_string.contains("$PSUDG,") && !get_string.contains(",ERR"))
	{
		for (int i = 0; i < config->get_PSU_count(); i++)
		{
			PSU_voltage[i] = QString(get_list.at(2+i*2)).toDouble();
			PSU_current[i] = QString(get_list.at(3+i*2)).toDouble();
			PSU_power[i] = PSU_voltage[i] * PSU_current[i];
			PSU_total_power += PSU_voltage[i] * PSU_current[i];
		}

		bool valid_readings = true;
		for (int i = 0; i < config->get_PSU_count(); i++)
		{
			if (PSU_voltage[i] < 0 || PSU_current[i] < 0)
			{
				valid_readings = false;
				break;
			}
		}

		//Avoid divide by zero...
		if (PSU_total_power > 0)
		{
			PSU_power_efficiency = convert_dbm_to_watt(PowerFRW_Value) / PSU_total_power * 100;
		}
		else
		{
			PSU_power_efficiency = 0;
		}

		// call the handlers to put stuff on the screen
		for (int i = 0; i < config->get_PSU_count(); i++)
		{
			handler_PSU_IU_get(i, PSU_voltage[i], PSU_current[i], PSU_power[i]);
		}

		handler_PSU_power_efficiency_get(PSU_power_efficiency);

		PSU_dissipation_watt = PSU_total_power - convert_dbm_to_watt(PowerFRW_Value) + convert_dbm_to_watt(PowerRFL_Value);
		handler_PSU_dissipation_get(PSU_dissipation_watt);
	}
	else
	{
		/* If PSU readings error out make that clear in the GUI by switching the all the parameter indications to '-' */
		ui->label_PSU1_voltage_3->setText("-");
		ui->label_PSU1_current_3->setText("-");
		ui->label_PSU1_power_3->setText("-");
		ui->label_PSU2_voltage_3->setText("-");
		ui->label_PSU2_current_3->setText("-");
		ui->label_PSU2_power_3->setText("-");
		ui->label_PSU3_voltage_3->setText("-");
		ui->label_PSU3_current_3->setText("-");
		ui->label_PSU3_power_3->setText("-");
		ui->label_PSU4_voltage_3->setText("-");
		ui->label_PSU4_current_3->setText("-");
		ui->label_PSU4_power_3->setText("-");
		ui->label_PSU_efficiency_5->setText("-");
		ui->label_PSU_dissipation_5->setText("-");
	}
}


void MainWindow::handler_PSU_IU_get(int psu_num, double voltage, double current, double power)
{
	switch (psu_num)
	{
		case 0:
			//PSU page
			ui->label_PSU1_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU1_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU1_power_3->setText(QString::number(power,'f',0));
			break;
		case 1:
			//PSU page
			ui->label_PSU2_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU2_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU2_power_3->setText(QString::number(power,'f',0));
			break;
		case 2:
			//PSU page
			ui->label_PSU3_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU3_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU3_power_3->setText(QString::number(power,'f',0));
			break;
		case 3:
			//PSU page
			ui->label_PSU4_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU4_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU4_power_3->setText(QString::number(power,'f',0));
			break;
		default:
			break;
	}
}

void MainWindow::handler_PSU_power_efficiency_get(double val)
{
	ui->label_PSU_efficiency_5->setText(QString::number(val, 'f',1));
}

void MainWindow::handler_PSU_voltage_setpoint_get(double val)
{
	if (val >= 0)
	{
		ui->pushButton_PSU_voltage_1->setText(zeroChopper(QString::number(val,'f',1)));
	}
	else
	{
		ui->pushButton_PSU_voltage_1->setText("-");
	}
}

void MainWindow::handler_PSU_current_limit_get(double val)
{
	if (val >= 0)
	{
		ui->pushButton_PSU_current_limit_1->setText(zeroChopper(QString::number(val,'f',1)));
	}
	else
	{
		ui->pushButton_PSU_current_limit_1->setText("-");
	}
}

void MainWindow::handler_PSU_enable_combined_get(bool enable)
{
	//
	//TODO:
	//All PSU's are shown as enabled; even though there may be less present.
	//

	QString enable_string = "";
	enable == true ? enable_string = "ON" : enable_string = "OFF";

	for(int i = 0; i < PSU_count; i++)
	{
		PSU_enable[i] = enable;
	}

	//PSU page
	ui->pushButton_PSU_enable_1->setText(enable_string);
	ui->pushButton_PSU_enable_1->setChecked(enable);

	ui->label_PSU1_enable_2->setText(enable_string);
	ui->label_PSU2_enable_2->setText(enable_string);
	ui->label_PSU3_enable_2->setText(enable_string);
	ui->label_PSU4_enable_2->setText(enable_string);
}

void MainWindow::handler_PSU_dissipation_get(double val)
{
	ui->label_PSU_dissipation_5->setText(zeroChopper(QString::number(val,'f',1)));
}


/**********************************************************************************************************************************************************************************
 * SERIAL PARSER
 * *******************************************************************************************************************************************************************************/
void MainWindow::show_warning_popup(QString status)
{
	QMessageBox message;
	message.warning(nullptr, "Communication Error", "Function returned Error:\n" + status);
}

void MainWindow::buffer_fillup_handler()
{
	//TODO: GET commands don't have these checks...
	QMessageBox message;
	message.warning(nullptr, "Read buffer fill-up", "Read Buffer fill-up detected!\nSoftware state may be compromised.n\nSoftware will shutdown");
	this->close();
}

bool MainWindow::serial_parser(QString status)
{
	bool success = true;

	/* TODO: GET commands don't have these checks... */

	/* theoretically it might be possible to run into the issue of rx filling up beyond the /r/n point, causing messages to be broken up.
	 * TX:	 "$PPDG,1"
	 * RX:	 "$PPDG,1,19.12035,20.93336\r\n$PP"
	 * TX:	 "$PPDG,1"
	 * RX:	 "DG,1,19.12035,21.00475\r\n$PPDG,1,19.09899,20.96905\r\n"
	 * This results in a broken message. worst case multiple messages could get broken up in a row.
	 * The piece of code below can at least detect when multiple message have been receive in one go, but doesn't do much to handle the situation yet... */

	QStringList status_list = status.split("\r\n");
	status_list.removeAll("");
	if(status_list.length() > 1)
	{
		qDebug() << "status:" << status_list << "status_list length: " << status_list.length();
		emit buffer_fillup_detected();
	}

	/* TODO: I should probably only process the first entry of the QStringlist, rather than the whole thing with possible extras attached... */

	if(!status.contains(",OK\r\n"))
	{
		show_warning_popup(status);
		if (status.contains("$ECS,"))
		{
			ECG_parser();
		}
		else if(status.contains("$FCS,"))
		{
			FCG_parser();
		}
		else if(status.contains("$PCS,"))
		{
			PCG_parser();
		}
		else if(status.contains("$ALCS,"))
		{
			ALL_ALCG_parser();
		}
		else if(status.contains("$ALES,"))
		{
			ALL_ALEG_parser();
		}
		else if(status.contains("$DLCS," ))
		{
			DLL_DLCG_parser();
		}
		else if(status.contains("$DLES," ))
		{
			DLL_DLEG_parser();
		}
		else if(status.contains("$DCS,"))
		{
			PWM_DCG_duty_cycle_parser();
		}
		else if(status.contains("$DCFS,"))
		{
			PWM_DCG_frequency_parser();
		}
		else if(status.contains("$DCTS,"))
		{
			PWM_DCG_PWM_trig_parser();
		}
		else if(status.contains("$FWDS," ))
		{
			FWDG_parser();
		}
		else if(status.contains("$RFLS," ))
		{
			RFLG_parser();
		}
		else if(status.contains("$PWRS," ))
		{
			PWRG_parser();
		}
		else if(status.contains("$PWRDS," ))
		{
			PWRDG_parser();
		}
		else if(status.contains("$PATS," ))
		{
			PATG_parser();
		}
		else if(status.contains("$CSS," ))
		{
			CSG_parser();
		}
		else if(status.contains("$RST,"))
		{
			QMessageBox message;
			message.critical(nullptr, "Reset failed", "Reset function did not return 'OK'.\nSG board status may be compromised.\nImmediate full system restart is recommended.");
		}
		else if(status.contains("$AGES,"))
		{
			QMessageBox message;
			message.critical(nullptr, "Autogain set failed", "Autogain function did not return 'OK'.\nSG board status may be compromised.\nImmediate full system restart is recommended.");
		}
		else if(status.contains("$AIS,"))
		{
			QMessageBox message;
			message.critical(nullptr, "Analog input enable failed", "Analog input function did not return 'OK'.\nSG board status may be compromised.\nImmediate full system restart is recommended.");
		}
		else if(status.contains("$GCS,"))
		{
			GCG_parser();
		}
		else if(status.contains("$MCS,"))
		{
			MCG_parser();
		}
		else if(status.contains("$PWRSGDS,"))
		{
			PWRSGDG_parser();
		}
		else if (status.contains("$PSUES,"))
		{
			PSUEG_parser();
		}
		else if (status.contains("$PSUVS,"))
		{
			PSUVG_parser();
		}
		else if (status.contains("$PSUIS,"))
		{
			PSUIG_parser();
		}
		else if (status.contains("$MCIES,"))
		{
			//No GET action available.
		}
		else
		{
			qDebug() << "Unknown Command";
		}
		success = false;
	}
	update_labels();
	return success;
}

//TODO: This function should is obsolete once DCG has been split up into multiple commands.
void MainWindow::PWM_DCG_parser()
{
	QString get_string = serial_getPWM_settings(SG_port,channel_select);
	QRegExp regexp("\\$DCG,\\d+,(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+).(\\d+),(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		PWM_freq_value = regexp.cap(1).toDouble();
		PWM_correction_factor = regexp.cap(2).toDouble();
		//target_PWM_trigger_mode = regexp.cap(3).toDouble();		//Gets should not be returning target values...
		PWM_trigger_mode = regexp.cap(3).toDouble();
		PWM_slave_port = regexp.cap(4).toInt();
		PWM_slave_pin = regexp.cap(5).toInt();
		PWM_master_port = regexp.cap(6).toInt();
		PWM_master_pin = regexp.cap(7).toInt();
		PWM_delay = (regexp.cap(8) + "." + regexp.cap(9)).toDouble();

		/* Duty Cycle is practically both a value and an ON/OFF switch for PWM.
		 * It requires some special handling to avoid DCG overwriting the software target value unnecessarily
		 * We have decided that the PWM value can only be set between 1 and 99%.
		 * 100% means PWM is disabled, 0% is for external triggering */

		double temp_DC_value = regexp.cap(10).toDouble();
		if (temp_DC_value == 100)
		{
			if (PWM_enabled == true)
			{
				ui->pushButton_PWM_enable_1->click();
			}
		}
		else if (temp_DC_value == 0)
		{
			/* enter external triggering mode if Duty Cycle is 0% */
			if (ext_trig_mode != 1)
			{
				ui->pushButton_external_triggering_ON_1->click();
			}
		}
		else
		{
			PWM_duty_cycle_value = temp_DC_value;
			if (PWM_enabled == false)
			{
				ui->pushButton_PWM_enable_1->click();
			}
		}

		//TODO: I don't remember why this is here, and there is probably a good reason for it, but it should also probably go away eventually...
		// Set the PWM Triggering correctly
		qDebug() << "DCG_parser activating PWM_triggering_mode_handler:";
		PWM_triggering_mode_handler();
		if (config->get_console_output_enabled() == true)
		{
			qDebug() << "PWM_freq_value " << PWM_freq_value;
			qDebug() << "PWM_correction_factor " << PWM_correction_factor;
			qDebug() << "PWM_trigger_mode " << PWM_trigger_mode;
			qDebug() << "PWM_slave_port " << PWM_slave_port;
			qDebug() << "PWM_slave_pin " << PWM_slave_pin;
			qDebug() << "PWM_master_port " << PWM_master_port;
			qDebug() << "PWM_master_pin " << PWM_master_pin;
			qDebug() << "PWM_delay " << PWM_delay;
			qDebug() << "PWM_duty_cycle_value " << temp_DC_value << " on SG board, but internally in GUI actually: " << PWM_duty_cycle_value << "\n";
		}
	}
}

void MainWindow::PWM_DCG_PWM_trig_parser()
{
	QString get_string = serial_getPWM_settings(SG_port,channel_select);
	QRegExp regexp("\\$DCG,\\d+,(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+).(\\d+),(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		PWM_trigger_mode = regexp.cap(3).toDouble();
		PWM_slave_port = regexp.cap(4).toInt();
		PWM_slave_pin = regexp.cap(5).toInt();
		PWM_master_port = regexp.cap(6).toInt();
		PWM_master_pin = regexp.cap(7).toInt();

		// Set the PWM Triggering correctly
		PWM_triggering_mode_handler();

		if (config->get_console_output_enabled() == true)
		{
			qDebug() << "PWM_trigger_mode " << PWM_trigger_mode;
			qDebug() << "PWM_slave_port " << PWM_slave_port;
			qDebug() << "PWM_slave_pin " << PWM_slave_pin;
			qDebug() << "PWM_master_port " << PWM_master_port;
			qDebug() << "PWM_master_pin " << PWM_master_pin;
		}
	}
}

void MainWindow::PWM_DCG_frequency_parser()
{
	QString get_string = serial_getPWM_settings(SG_port,channel_select);
	QRegExp regexp("\\$DCG,\\d+,(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+).(\\d+),(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		PWM_freq_value = regexp.cap(1).toDouble();
		PWM_correction_factor = regexp.cap(2).toDouble();
		PWM_delay = (regexp.cap(8) + "." + regexp.cap(9)).toDouble();

		if (config->get_console_output_enabled() == true)
		{
			qDebug() << "PWM_freq_value " << PWM_freq_value;
			qDebug() << "PWM_correction_factor " << PWM_correction_factor;
			qDebug() << "PWM_delay " << PWM_delay;
		}
	}
}

void MainWindow::PWM_DCG_duty_cycle_parser()
{
	QString get_string = serial_getPWM_settings(SG_port,channel_select);
	QRegExp regexp("\\$DCG,\\d+,(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+).(\\d+),(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		/* Duty Cycle is practically both a value and an ON/OFF switch for PWM.
		 * It requires some special handling to avoid DCG overwriting the software target value unnecessarily
		 * We have decided that the PWM value can only be set between 1 and 99%.
		 * 100% means PWM is disabled, 0% is for external triggering */

		double temp_DC_value = regexp.cap(10).toDouble();
		if (temp_DC_value == 100)
		{
			if (ui->pushButton_PWM_enable_1->isChecked() || PWM_enabled == true)
			{
				ui->pushButton_PWM_enable_1->setChecked(false);
				on_pushButton_PWM_enable_1_clicked(false);
			}

			if (!ui->pushButton_external_triggering_OFF_1->isChecked())
			{
				ui->pushButton_external_triggering_OFF_1->setChecked(true);
				on_pushButton_external_triggering_OFF_1_clicked();
			}
		}
		else if (temp_DC_value == 0)
		{
			if (ui->pushButton_PWM_enable_1->isChecked() || PWM_enabled == true)
			{
				ui->pushButton_PWM_enable_1->setChecked(false);
				on_pushButton_PWM_enable_1_clicked(false);
			}

			/* enter external triggering mode if Duty Cycle is 0% */
			if (!ui->pushButton_external_triggering_ON_1->isChecked())
			{
				ui->pushButton_external_triggering_ON_1->setChecked(true);
				on_pushButton_external_triggering_ON_1_clicked();
			}
		}
		else
		{
			PWM_duty_cycle_value = temp_DC_value;

			if (PWM_enabled == false || !ui->pushButton_PWM_enable_1->isChecked())
			{
				ui->pushButton_PWM_enable_1->setChecked(true);
				on_pushButton_PWM_enable_1_clicked(true);
			}

			if (!ui->pushButton_external_triggering_OFF_1->isChecked())
			{
				ui->pushButton_external_triggering_OFF_1->setChecked(true);
				on_pushButton_external_triggering_OFF_1_clicked();
			}
		}

//		External_triggering_handler();

		if (config->get_console_output_enabled() == true)
		{
			qDebug() << "PWM_duty_cycle_value " << temp_DC_value << " on SG board, but internally in GUI actually: " << PWM_duty_cycle_value << "\n";
		}
	}
}

void MainWindow::DLL_DLCG_parser()
{
	QString get_string = serial_getDLL_settings(SG_port,channel_select);
	QRegExp regexp("\\$DLCG,\\d+,(\\d+).(\\d+),(\\d+).(\\d+),(\\d+).(\\d+),(\\d+).(\\d+),(\\d+).(\\d+),(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		DLL_freqLimit_DOWN_value = (regexp.cap(1) + "." + regexp.cap(2)).toDouble();
		DLL_freqLimit_UP_value = (regexp.cap(3) + "." + regexp.cap(4)).toDouble();
		DLL_freq_start_value = (regexp.cap(5) + "." + regexp.cap(6)).toDouble();
		DLL_freq_step_value = (regexp.cap(7) + "." + regexp.cap(8)).toDouble();
		DLL_threshold_value = (regexp.cap(9) + "." + regexp.cap(10)).toDouble();
		DLL_main_delay_value = regexp.cap(11).toDouble();
		if (config->get_console_output_enabled() == true)
		{
			qDebug() << "DLL_freqLimit_DOWN_value " << DLL_freqLimit_DOWN_value;
			qDebug() << "DLL_freqLimit_UP_value " << DLL_freqLimit_UP_value;
			qDebug() << "DLL_freq_start_value " << DLL_freq_start_value;
			qDebug() << "DLL_freq_step_value " << DLL_freq_step_value;
			qDebug() << "DLL_threshold_value " << DLL_threshold_value;
			qDebug() << "DLL_main_delay_value" << DLL_main_delay_value << "\n";
		}
	}
}

void MainWindow::DLL_DLEG_parser()
{
	QString get_string = serial_getDLL_status(SG_port,channel_select);
	QRegExp regexp("\\$DLEG,\\d+,(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		DLL_enabled = (bool)regexp.cap(1).toInt();
		//DLL_mode_enable(DLL_enabled);
		qDebug() << "DLL_enabled" << DLL_enabled << "\n";
	}
}

void MainWindow::ALL_ALCG_parser()
{
	QString get_string = serial_getALL_settings(SG_port,channel_select);
	QRegExp regexp("\\$ALCG,\\d+,(\\d+).(\\d+),(\\d+).(\\d+),(\\d+).(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		// upper and lower limits are turned around. this issue exists in firmware as well, inconsistent with DLL should probably be fixed.
		ALL_freqLimit_DOWN_value = (regexp.cap(1) + "." + regexp.cap(2)).toDouble();
		ALL_freqLimit_UP_value = (regexp.cap(3) + "." + regexp.cap(4)).toDouble();
		ALL_threshold_value = (regexp.cap(5) + "." + regexp.cap(6)).toDouble();
		if (config->get_console_output_enabled() == true)
		{
			qDebug() << "ALL_freqLimit_DOWN_value" << ALL_freqLimit_DOWN_value;
			qDebug() << "ALL_freqLimit_UP_value" << ALL_freqLimit_UP_value;
			qDebug() << "ALL_threshold_value" << ALL_threshold_value << "\n";
		}
	}
}

void MainWindow::ALL_ALEG_parser()
{
	QString get_string = serial_getALL_status(SG_port,channel_select);
	QRegExp regexp("\\$ALEG,\\d+,(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		ALL_enabled = (bool)regexp.cap(1).toInt();
		//ALL_mode_enable(ALL_enabled);
		qDebug() << "ALL_enabled" << ALL_enabled << "\n";
	}
}

void MainWindow::FCG_parser(bool console_output_enabled)
{
	QString get_string = serial_getFrequency(SG_port,channel_select,console_output_enabled);
	QRegExp regexp("\\$FCG,\\d+,(\\d+).(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		freq_value = (regexp.cap(1) + "." + regexp.cap(2)).toDouble() * 1000000;
		if (console_output_enabled == true)
		{
			qDebug() << "freq_value" << QString::number(freq_value,'f',3) << "\n";
		}
	}
}

void MainWindow::PCG_parser()
{
	QString get_string = serial_getPhase(SG_port,channel_select);
	QRegExp regexp("\\$PCG,\\d+,(\\d+).(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		phase_value = (regexp.cap(1) + "." + regexp.cap(2)).toDouble();
		qDebug() << "phase_value" << phase_value << "\n";
	}
}

void MainWindow::ECG_parser(bool console_output_enabled)
{
	QString get_string = serial_getRF_status(SG_port,channel_select, console_output_enabled);
	QRegExp regexp("\\$ECG,\\d+,(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		RF_enabled = (bool)regexp.cap(1).toInt();
		if (console_output_enabled == true)
		{
			qDebug() << "RF_enabled" << RF_enabled << "\n";
		}
	}
}

void MainWindow::GCG_parser(bool console_output_enabled)
{
	QString get_string = serial_getVGA_attenuation(SG_port,channel_select,console_output_enabled);
	QRegExp regexp("\\$GCG,\\d+,(\\d+).(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		QString temp_val = regexp.cap(1) + "." + regexp.cap(2);
		AIS_attenuation_value = temp_val.toDouble();
		VGA_attenuation_value = temp_val.toDouble();
		if (console_output_enabled == true)
		{
			qDebug() << "VGA attenuation value: " << VGA_attenuation_value << "\n";
		}
	}
}

void MainWindow::MCG_parser(bool console_output_enabled)
{
	QString get_string = serial_getMagnitude(SG_port,channel_select,console_output_enabled);
	QRegExp regexp("\\$MCG,\\d+,(\\d+).(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		QString temp_val = regexp.cap(1) + "." + regexp.cap(2);
		magnitude_value = temp_val.toDouble();
		if (console_output_enabled == true)
		{
			qDebug() << "Magnitude value: " << magnitude_value << "\n";
		}
	}
}

void MainWindow::PWRSGDG_parser()
{
	QString get_string = serial_getSGxPower_dbm(SG_port,channel_select);
	QRegExp regexp("\\$PWRSGDG,\\d+,(\\d+).(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		QString temp_val = regexp.cap(1) + "." + regexp.cap(2);
		sgx_power_value = temp_val.toDouble();
		qDebug() << "SGx Power value:" << sgx_power_value << "\n";
	}
}

void MainWindow::PTG_parser(bool console_output_enabled)
{
	QString get_string = serial_getTemperature(SG_port,channel_select, console_output_enabled);
	if (get_string.contains("$PTG,") && get_string.contains("\r\n") && !get_string.contains("ERR"))
	{
		QStringList get_list = get_string.split(",");
		Temperature_Value = get_list.at(2).toDouble();

		if (console_output_enabled)
		{
			qDebug() << "Temperature_Value: " << Temperature_Value << "\n";
		}
	}
}

void MainWindow::PWRG_parser()
{
	QString get_string = serial_getPower_watt(SG_port,channel_select);
	QRegExp regexp("\\$PWRG,\\d+,(\\d+).(\\d+\r\n)");

	if (regexp.indexIn(get_string)>=0)
	{
		power_watt_value = (regexp.cap(1) + "." + regexp.cap(2)).toDouble();
		power_dbm_value = zeroChopper(QString::number(convert_watt_to_dbm(power_watt_value),'f',6)).toDouble();
		qDebug() << "power_watt_value" << power_watt_value;
		qDebug() << "power_dbm_value" << power_dbm_value << "\n";

		/* PA Type 1 uses PWRDS for standalone feedforward control, so PWRDG is needed */
		if (PA_type == 1)
		{
			power_dbm_standalone_value = zeroChopper(QString::number(convert_watt_to_dbm(power_watt_value),'f',6)).toDouble();
		}
	}
}

void MainWindow::PWRDG_parser()
{
	QString get_string = serial_getPower_dbm(SG_port,channel_select);
	QRegExp regexp("\\$PWRDG,\\d+,(\\d+).(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		power_dbm_value = (regexp.cap(1) + "." + regexp.cap(2)).toDouble();
		power_watt_value = zeroChopper(QString::number(convert_dbm_to_watt(power_dbm_value),'f',1)).toDouble();
		qDebug() << "power_dbm_value" << power_dbm_value;
		qDebug() << "power_watt_value" << power_watt_value;

		// In PATS 1 feed forward mode $PWRDS/G is also used, but it has different limits.
		if (PA_type == 1)
		{
			power_dbm_standalone_value = (regexp.cap(1) + "." + regexp.cap(2)).toDouble();
			qDebug() << "power_dbm_standalone_value" << power_dbm_standalone_value << "\n";
		}

		qDebug() << "\n";
	}
}

void MainWindow::FWDG_parser()
{
	if (config->get_support_DVGA() == true)
	{
		QString get_string = serial_get_DVGA_forward(SG_port,channel_select);
		QRegExp regexp("\\$FWDG,\\d+,(\\d+),(\\d+).(\\d+)\r\n");
		if (ALL_enabled == true)
		{
			if (regexp.indexIn(get_string)>=0)
			{
				DVGA_amplifier_forward_enabled = (bool)regexp.cap(1).toInt();
				DVGA_attenuation_forward = (regexp.cap(2) + "." + regexp.cap(3)).toDouble();
				qDebug() << "DVGA_amplifier_forward_enabled" << DVGA_amplifier_forward_enabled;
				qDebug() << "DVGA_attenuation_forward" << DVGA_attenuation_forward << "\n";
			}
		}
	}
}

void MainWindow::RFLG_parser()
{
	if (config->get_support_DVGA() == true)
	{
		QString get_string = serial_get_DVGA_reflected(SG_port,channel_select);
		QRegExp regexp("\\$RFLG,\\d+,(\\d+),(\\d+).(\\d+)\r\n");
		if (ALL_enabled == true)
		{
			if (regexp.indexIn(get_string)>=0)
			{
				DVGA_amplifier_reflected_enabled = (bool)regexp.cap(1).toInt();
				DVGA_attenuation_reflected = (regexp.cap(2) + "." + regexp.cap(3)).toDouble();
				qDebug() << "DVGA_amplifier_reflected_enabled" << DVGA_amplifier_reflected_enabled;
				qDebug() << "DVGA_attenuation_reflected" << DVGA_attenuation_reflected << "\n";
			}
		}
	}
}

void MainWindow::PATG_parser()
{
	QString get_string = serial_get_PA_type(SG_port,channel_select);
	QRegExp regexp("\\$PATG,\\d+,(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		PA_type = regexp.cap(1).toInt();
		qDebug() << "PA_type" << PA_type << "\n";
	}
}

void MainWindow::CSG_parser()
{
	//Clock Source Get
	QString get_string = serial_get_clock_source(SG_port,channel_select);
	QRegExp regexp("\\$CSG,\\d+,(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		clock_source = regexp.cap(1).toInt();
		qDebug() << "clock_source" << clock_source << "\n";
	}
}

void MainWindow::ST_parser(bool console_output_enabled)
{
	QString get_string = serial_getStatus(SG_port, channel_select, console_output_enabled);
	QRegExp regexp;
	regexp.setCaseSensitivity(Qt::CaseInsensitive);
	regexp.setPattern("\\$ST,\\d+,(\\S+),(\\S+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		SG_status = regexp.cap(2).toULongLong(nullptr,16);
		if (console_output_enabled == true)
		{
			qDebug() << "Error status: " << QString("0x%1").arg(SG_status,0,16) << "\n";
		}
	}
}

void MainWindow::VER_parser()
{
	QString get_string = serial_getVersion(SG_port, channel_select);
	if (get_string.contains("$VER,") && get_string.contains("\r\n"))
	{
		QStringList get_list = get_string.split(",");
		firmware_version_actual[0] = get_list.at(3).toInt();
		firmware_version_actual[1] = get_list.at(4).toInt();
		firmware_version_actual[2] = get_list.at(5).toInt();
		firmware_version_actual[3] = get_list.at(6).toInt();

		SG_firmware_version = QString::number(firmware_version_actual[0]) + "." + QString::number(firmware_version_actual[1]) + "." +QString::number(firmware_version_actual[2]);
		if (firmware_version_actual[3] > 0)
		{
			SG_firmware_version += ("." + QString::number(firmware_version_actual[3]));
		}
	}
	else
	{
		get_string.replace(0,8,"");
		get_string.chop(2);
		SG_firmware_version = get_string;
	}

	qDebug() << "Firmware version: " << SG_firmware_version << "\n";
}

/* Get the PSU enable state */
void MainWindow::PSUEG_parser()
{
	QString get_string = serial_getPSU_enable(SG_port, channel_select);
	QStringList get_list = get_string.split(",");

	if (get_string.contains("$PSUEG,") && get_string.contains("\r\n") && !get_string.contains(",ERR"))
	{
		PSU_enable_combined = (bool)QString(get_list.at(2)).toInt();
		handler_PSU_enable_combined_get(PSU_enable_combined);
	}
	else
	{
		/* This code will make the PSU enable labels show 'OFF' whenever the PSU GPIO button is unpressed, instead of just a dash for invalid readings.
		 * The code works fine, I've just decided against having it active as it's a very minor feature and as I suspect it won't scale very nicely with other systems that may exist in the future.
		 * I also wasn't entirely sure whether I liked the functionality or not. */
//		#if defined(Q_OS_LINUX)
//		if (config->get_support_HAT_B14_0835())
//		{
//			if (GPIO_PSU_button_enable->readValue() < 1)
//			{
//				PSU_enable_combined = false;
//				handler_PSU_enable_combined_get(PSU_enable_combined);
//				return;
//			}
//		}
//		#endif

		qDebug() << "PSUEG_parser: PSU enable get failure" << "\n";
		ui->pushButton_PSU_enable_1->setText("-");
		ui->label_PSU1_enable_2->setText("-");
		ui->label_PSU2_enable_2->setText("-");
		ui->label_PSU3_enable_2->setText("-");
		ui->label_PSU4_enable_2->setText("-");
	}
}

/* Get the PSU voltage setpoint */
void MainWindow::PSUVG_parser()
{
	QString get_string = serial_getPSU_voltage_setpoint(SG_port, channel_select);
	QStringList get_list = get_string.split(",");

	if (get_string.contains("$PSUVG,") && get_string.contains("\r\n") && !get_string.contains(",ERR"))
	{
		PSU_voltage_setpoint = QString(get_list.at(2)).toDouble();
		handler_PSU_voltage_setpoint_get(PSU_voltage_setpoint);
	}
	else
	{
		ui->pushButton_PSU_voltage_1->setText("-");
	}

	qDebug() << "PSU voltage setpoint: " << PSU_voltage_setpoint << "\n";
}

/* Get the PSU current limit of all PSUs combined */
void MainWindow::PSUIG_parser()
{
	QString get_string = serial_getPSU_current_limit(SG_port, channel_select);
	QStringList get_list = get_string.split(",");

	if (get_string.contains("$PSUIG,") && get_string.contains("\r\n") && !get_string.contains(",ERR"))
	{
		PSU_current_limit = QString(get_list.at(2)).toDouble();
		handler_PSU_current_limit_get(PSU_current_limit);
	}
	else
	{
		ui->pushButton_PSU_current_limit_1->setText("-");
	}

	qDebug() << "PSU current limit: " << PSU_current_limit << "\n";
}


/**********************************************************************************************************************************************************************************
 * NOTIFICATION / WARNING BAR
 * *******************************************************************************************************************************************************************************/

void MainWindow::show_notification(QString message, int time)
{
	notification_timer->setSingleShot(true);
	connect(notification_timer, SIGNAL(timeout()), this, SLOT(close_notification()));
	ui->label_notification->setText(message);
	ui->label_notification->setVisible(true);
	notification_timer->start(time);
}

void MainWindow::close_notification()
{
	ui->label_notification->setVisible(false);
}

void MainWindow::show_warning(QString message)
{
	ui->label_error_message->setText(message);
	ui->label_error_message->setVisible(true);
}

void MainWindow::close_warning()
{
	ui->label_error_message->setVisible(false);
}

/**********************************************************************************************************************************************************************************
 * VERSION COMPATIBILITY CHECK
 * *******************************************************************************************************************************************************************************/
void MainWindow::version_control()
{
	if(isVersionCompatible(firmware_version_actual,firmware_version_requirement) == false)
	{
		QString comp_warning =	"The current firmware version is unsupported by this application and may result in unpredictable behaviour. "
								"Please upgrade the firmware of your SGx board to at least the recommended version.\n"
								"\nRecommended firmware: v";
		for (int i = 0; i < sizeof(&firmware_version_requirement); i++)
		{
			comp_warning += QString::number(firmware_version_requirement[i]) + ".";
		}
		comp_warning.chop(1);	//chop the redundant period
		comp_warning += "\nCurrent firmware: " + SG_firmware_version +
						"\n\nProceed at own risk.";

		QMessageBox message;
		message.warning(nullptr,	"Software imcompatibility", comp_warning);
		show_notification("Firmware is outdated!\nProceed at own risk.");
	}
}

/**********************************************************************************************************************************************************************************
 * HELP
 * *******************************************************************************************************************************************************************************/
void MainWindow::on_listWidget_currentItemChanged(QListWidgetItem *current, QListWidgetItem *previous)
{
	QFile file;

	if (current == &listWidgetItem_system)
	{
		ui->textBrowser->setHtml(	"<body style=\" font-family:'Arial';\">"
									"<b>Toolkit GUI version:</b><br>"	+ toolkit_GUI_version	+ "<br>" + "<br>" +
									"<b>SSG firmware version:</b><br>"	+ SG_firmware_version	+ "<br>" + "<br>" +
									"<b>SSG manufacturer:</b><br>"		+ SG_manufacturer		+ "<br>" + "<br>" +
									"<b>SSG model:</b><br>"				+ SG_board_model		+ "<br>" + "<br>" +
									"<b>SSG serial number:</b><br>"		+ SG_serial_number		+ "<br>" + "<br>" +
									"<b>PA type:</b><br>"				+ QString::number(PA_type) +
									"</body>");
/*		ui->textBrowser->setText(
									"Toolkit GUI version:\n"	+ toolkit_GUI_version	+ "\n" + "\n" +
									"SSG firmware version:\n"	+ SG_firmware_version	+ "\n" + "\n" +
									"SSG manufacturer:\n"		+ SG_manufacturer		+ "\n" + "\n" +
									"SSG model:\n"				+ SG_board_model		+ "\n" + "\n" +
									"SSG serial number:\n"		+ SG_serial_number		+ "\n" + "\n" +
									"PA type:\n"				+ QString::number(PA_type)
								); */

		return;		//Exit function; There is no Qfile to open in this case;
	}
	else if (current == &listWidgetItem_license)
	{
		file.setFileName(":/copyrights/GNU_GPLv3_License.html");
	}
	else if (current == &listWidgetItem_about)
	{
		file.setFileName(":/copyrights/about.html");
	}
	else if (current == &listWidgetItem_copyright)
	{
		file.setFileName(":/copyrights/pinkRF_copyright.html");
	}
	else
	{
		file.setFileName("");
	}


	if (file.open(QFile::ReadOnly | QFile::Text))
	{
		QTextStream stream(&file);
		ui->textBrowser->setHtml(file.readAll());
	}
	else
	{
		ui->textBrowser->clear();
	}
}
